<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\sanitize_placeholder\Service\ThematicFaker;

/**
 * Generates institution / organisation names for placeholder sanitization.
 */
final class InstitutionStrategy implements StrategyInterface {

  /**
   * Construct the strategy.
   *
   * @param \Drupal\sanitize_placeholder\Service\ThematicFaker $faker
   *   The faker wrapper used to generate deterministic sample data.
   */
  public function __construct(
    private readonly ThematicFaker $faker,
  ) {}

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'institution';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return 'Institution/Company';
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $field): string {
    // Use deterministic, algorithmic company names even without fakerphp/faker.
    $seed = $this->seedFrom($entity, $field);
    return $this->faker->company($seed);
  }

  /**
   * Build a stable seed for deterministic output per row/field.
   */
  private function seedFrom(EntityInterface $entity, FieldDefinitionInterface $field): string {
    $idPart = method_exists($entity, 'uuid') && $entity->uuid()
      ? $entity->uuid()
      : (string) $entity->id();
    return $entity->getEntityTypeId() . ':' . $idPart . ':' . $field->getName();
  }

}
