<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder_extra\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for extra placeholder strategies.
 */
final class ExtraSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sanitize_placeholder_extra_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['sanitize_placeholder_extra.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('sanitize_placeholder_extra.settings');

    // Domain strategy options (used by DomainStrategy).
    $form['domain'] = [
      '#type' => 'details',
      '#title' => $this->t('Domain strategy'),
      '#open' => TRUE,
    ];

    $form['domain']['domain_tlds'] = [
      '#type' => 'textfield',
      '#title' => $this->t('TLD allowlist'),
      '#description' => $this->t('Comma-separated list, for example: %ex', ['%ex' => 'com, org, eu, fr, pt']),
      '#default_value' => implode(', ', (array) ($config->get('domain.tlds') ?? [])),
    ];

    $form['domain']['domain_words'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed words'),
      '#description' => $this->t('One word per line. Words are combined to produce human-friendlier domains.'),
      '#default_value' => implode("\n", (array) ($config->get('domain.words') ?? [])),
      '#rows' => 6,
    ];

    // Pattern strategy (maps to top-level "patterns" array in config).
    $form['patterns'] = [
      '#type' => 'details',
      '#title' => $this->t('Patterns'),
      '#open' => TRUE,
    ];

    $existing_patterns = (array) ($config->get('patterns') ?? []);
    $form['patterns']['patterns_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Pattern list'),
      '#description' => $this->t('One pattern per line, for example: %a or %b', [
        '%a' => '{letters:3}-{digits:4}',
        '%b' => '{company}-{digits:3}',
      ]),
      '#default_value' => implode("\n", $existing_patterns),
      '#rows' => 6,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Normalize TLDs.
    $tlds = array_filter(array_map('trim', explode(',', (string) $form_state->getValue('domain_tlds'))));
    $form_state->setValue('domain_tlds', implode(', ', $tlds));

    // Normalize domain words (one per line).
    $words_text = (string) $form_state->getValue('domain_words');
    $words = preg_split('/\R/u', $words_text, -1, PREG_SPLIT_NO_EMPTY) ?: [];
    $words = array_values(array_map('trim', $words));
    $form_state->setValue('domain_words', implode("\n", $words));

    // Clean pattern lines (keep order, drop empties).
    $patterns_text = (string) $form_state->getValue('patterns_list');
    $patterns = preg_split('/\R/u', $patterns_text, -1, PREG_SPLIT_NO_EMPTY) ?: [];
    $patterns = array_values(array_map('trim', $patterns));
    $form_state->setValue('patterns_list', implode("\n", $patterns));

    // Only persist if there are no validation errors.
    if (empty($form_state->getErrors())) {
      // Re-read normalized values from the form state for saving.
      $tlds = array_filter(array_map('trim', explode(',', (string) $form_state->getValue('domain_tlds'))));
      $words = preg_split('/\R/u', (string) $form_state->getValue('domain_words'), -1, PREG_SPLIT_NO_EMPTY) ?: [];
      $words = array_values(array_map('trim', $words));
      $patterns = preg_split('/\R/u', (string) $form_state->getValue('patterns_list'), -1, PREG_SPLIT_NO_EMPTY) ?: [];
      $patterns = array_values(array_map('trim', $patterns));

      $this->configFactory->getEditable('sanitize_placeholder_extra.settings')
        ->set('domain.tlds', $tlds)
        ->set('domain.words', $words)
        ->set('patterns', $patterns)
        ->save();
    }

    parent::validateForm($form, $form_state);
  }

}
