<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sanitize_placeholder\Strategy\StrategyInterface;

/**
 * Generates a German IBAN (country code DE).
 *
 * Format: DEkk + 18 digits (total length 22).
 * The check digits (kk) are computed using the standard IBAN mod97 algorithm.
 */
final class IbanDeStrategy implements StrategyInterface {

  use StringTranslationTrait;

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'iban_de';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return (string) $this->t('German IBAN (DE)');
  }

  /**
   * {@inheritdoc}
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $fieldDefinition): string {
    // German BBAN is 18 numeric characters.
    $bban = $this->randomDigits(18);
    $check = $this->ibanCheckDigits('DE', $bban);
    return 'DE' . $check . $bban;
  }

  /**
   * Create a string of N random digits (0-9).
   *
   * @throws \Exception
   */
  private function randomDigits(int $length): string {
    $out = '';
    for ($i = 0; $i < $length; $i++) {
      $out .= random_int(0, 9);
    }
    return $out;
  }

  /**
   * Compute IBAN check digits for given country and BBAN.
   */
  private function ibanCheckDigits(string $country, string $bban): string {
    $country = mb_strtoupper($country, 'UTF-8');
    $rearranged = $bban . $country . '00';
    $numeric = $this->alnumToIbanNumber($rearranged);
    $remainder = $this->mod97($numeric);
    $check = 98 - $remainder;
    return str_pad((string) $check, 2, '0', STR_PAD_LEFT);
  }

  /**
   * Convert alphanumeric IBAN string to its numeric representation.
   */
  private function alnumToIbanNumber(string $value): string {
    $value = mb_strtoupper($value, 'UTF-8');
    $out = '';
    $len = mb_strlen($value, 'UTF-8');
    for ($i = 0; $i < $len; $i++) {
      $ch = mb_substr($value, $i, 1, 'UTF-8');
      $ord = ord($ch);
      if ($ord >= 48 && $ord <= 57) {
        $out .= $ch;
      }
      else {
        $out .= ord($ch) - 55;
      }
    }
    return $out;
  }

  /**
   * The mod 97 for very long numeric strings.
   */
  private function mod97(string $numeric): int {
    $remainder = 0;
    $len = strlen($numeric);
    for ($i = 0; $i < $len; $i++) {
      $chunk = $remainder . $numeric[$i];
      $remainder = (int) $chunk % 97;
    }
    return $remainder;
  }

}
