<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sanitize_placeholder\Strategy\StrategyInterface;

/**
 * Generates a Portuguese IBAN (country code PT).
 *
 * Format: PTkk + 21 digits (total length 25).
 * The check digits (kk) are computed using the standard IBAN mod97 algorithm.
 */
final class IbanPtStrategy implements StrategyInterface {

  use StringTranslationTrait;

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'iban_pt';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return (string) $this->t('Portuguese IBAN (PT)');
  }

  /**
   * {@inheritdoc}
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $fieldDefinition): string {
    // Portuguese BBAN is 21 numeric characters.
    $bban = $this->randomDigits(21);
    $check = $this->ibanCheckDigits('PT', $bban);
    return 'PT' . $check . $bban;
  }

  /**
   * Create a string of N random digits (0-9).
   *
   * @throws \Exception
   */
  private function randomDigits(int $length): string {
    $out = '';
    for ($i = 0; $i < $length; $i++) {
      $out .= random_int(0, 9);
    }
    return $out;
  }

  /**
   * Compute IBAN check digits for given country and BBAN.
   *
   * @param string $country
   *   Two-letter ISO country code.
   * @param string $bban
   *   The BBAN (digits / uppercase letters).
   *
   * @return string
   *   Two-digit checksum string.
   */
  private function ibanCheckDigits(string $country, string $bban): string {
    // Prepare "BBAN + COUNTRY + 00", letters as numbers (A=10..Z=35).
    $country = mb_strtoupper($country, 'UTF-8');
    $rearranged = $bban . $country . '00';
    $numeric = $this->alnumToIbanNumber($rearranged);
    // Compute 98 - (numeric mod 97).
    $remainder = $this->mod97($numeric);
    $check = 98 - $remainder;
    return str_pad((string) $check, 2, '0', STR_PAD_LEFT);
  }

  /**
   * Convert alphanumeric IBAN string to its numeric representation.
   */
  private function alnumToIbanNumber(string $value): string {
    $value = mb_strtoupper($value, 'UTF-8');
    $out = '';
    $len = mb_strlen($value, 'UTF-8');
    for ($i = 0; $i < $len; $i++) {
      $ch = mb_substr($value, $i, 1, 'UTF-8');
      $ord = ord($ch);
      if ($ord >= 48 && $ord <= 57) {
        // 0-9.
        $out .= $ch;
      }
      else {
        // A=10 .. Z=35.
        $out .= ord($ch) - 55;
      }
    }
    return $out;
  }

  /**
   * The mod 97 for very long numeric strings.
   */
  private function mod97(string $numeric): int {
    $len = strlen($numeric);
    $remainder = 0;
    for ($i = 0; $i < $len; $i++) {
      $chunk = $remainder . $numeric[$i];
      $remainder = (int) $chunk % 97;
    }
    return $remainder;
  }

}
