<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\sanitize_placeholder\Strategy\StrategyInterface;

/**
 * Generates a Spanish NIF/NIE.
 *
 * - NIF: 8 digits + control letter.
 * - NIE: prefix (X/Y/Z) + 7 digits + control letter.
 * Control letter uses the canonical sequence by number % 23.
 */
final class NifEsStrategy implements StrategyInterface {

  use StringTranslationTrait;

  /**
   * Map for control letter calculation (index by number % 23).
   *
   * @var string
   */
  private const CONTROL = 'TRWAGMYFPDXBNJZSQVHLCKE';

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'nif_es';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return (string) $this->t('Spanish NIF/NIE');
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $fieldDefinition): string {
    // Randomly choose NIF (70%) or NIE (30%).
    $isNie = (random_int(1, 100) <= 30);

    if ($isNie) {
      $prefix = ['X', 'Y', 'Z'][random_int(0, 2)];
      $digits = $this->randomDigits(7);
      $numeric = $this->nieToNumber($prefix, $digits);
      $letter = $this->controlLetter($numeric);
      return $prefix . $digits . $letter;
    }

    // NIF: 8 digits + letter.
    $digits = $this->randomDigits(8);
    $letter = $this->controlLetter((int) $digits);
    return $digits . $letter;
  }

  /**
   * Create a string of N random digits.
   *
   * @throws \Exception
   */
  private function randomDigits(int $length): string {
    $out = '';
    for ($i = 0; $i < $length; $i++) {
      $out .= random_int(0, 9);
    }
    return $out;
  }

  /**
   * Convert a NIE prefix + digits into its numeric value for control calc.
   */
  private function nieToNumber(string $prefix, string $digits): int {
    $prefix = mb_strtoupper($prefix, 'UTF-8');
    // X->0, Y->1, Z->2.
    $map = ['X' => '0', 'Y' => '1', 'Z' => '2'];
    $first = $map[$prefix] ?? '0';
    return (int) ($first . $digits);
  }

  /**
   * Calculate the control letter by (number % 23).
   */
  private function controlLetter(int $number): string {
    $idx = $number % 23;
    return mb_substr(self::CONTROL, $idx, 1, 'UTF-8');
  }

}
