<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\sanitize_placeholder\Service\ThematicFaker;
use Drupal\sanitize_placeholder\Strategy\StrategyManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure sanitize_placeholder settings.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Strategy manager service.
   */
  protected StrategyManager $strategyManager;

  /**
   * Faker wrapper (optional faker, with algorithmic fallbacks).
   *
   * @var \Drupal\sanitize_placeholder\Service\ThematicFaker
   */
  protected ThematicFaker $thematicFaker;

  /**
   * Construct the settings form.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StrategyManager $strategy_manager,
    TypedConfigManagerInterface $typed_config_manager,
    ThematicFaker $thematic_faker,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->strategyManager = $strategy_manager;
    $this->thematicFaker = $thematic_faker;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('sanitize_placeholder.strategy_manager'),
      $container->get('config.typed'),
      $container->get('sanitize_placeholder.thematic_faker'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['sanitize_placeholder.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'sanitize_placeholder_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('sanitize_placeholder.settings');
    // Safe check: use the service if present, otherwise detect Faker directly.
    $fakerAvailable = (
      property_exists($this, 'thematicFaker')
      && method_exists($this->thematicFaker, 'hasFaker')
    )
      ? $this->thematicFaker->hasFaker()
      : (class_exists('\\Faker\\Factory') && interface_exists('\\Faker\\Generator'));

    // General.
    $form['username_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Max username length'),
      '#default_value' => (int) ($config->get('username_max_length') ?? 10),
      '#min' => 1,
      '#description' => $this->t('Maximum characters for generated usernames.'),
    ];

    $form['faker_locale'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Faker locale'),
      '#default_value' => (string) ($config->get('faker_locale') ?? ''),
      '#description' => $this->t('Optional Faker locale code (e.g. "fr_FR", "pt_PT"). Leave empty for default (en_US).'),
      '#disabled' => !$fakerAvailable,
    ];

    $form['deterministic'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Deterministic replacements'),
      '#default_value' => (bool) ($config->get('deterministic') ?? TRUE),
      '#description' => $this->t('When enabled, seeding will make results repeatable.'),
    ];

    // Rules textarea: entity.bundle.field=strategy (one per line).
    $form['replacements'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Per-field replacement rules'),
      '#default_value' => $this->stringifyRules((array) ($config->get('replacements') ?? [])),
      '#rows' => 8,
      '#description' => $this->t('Format: <code>entity.bundle.field=strategy</code> — one per line. See the "Available strategies" list below for valid strategy IDs. After <code>drush sql:sanitize</code> finishes, this module applies replacements only to fields detected as sanitized; running <code>drush sp:fake</code> applies the configured replacements to all targeted fields (or those limited via options).'),
    ];

    // Post-hook behavior after sql:sanitize.
    $form['post_hook'] = [
      '#type' => 'details',
      '#title' => $this->t('Run after sql:sanitize'),
      '#open' => FALSE,
    ];
    $form['post_hook']['post_hook_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Limit per rule'),
      '#min' => 1,
      '#default_value' => (int) ($config->get('post_hook_limit') ?? 5000),
      '#description' => $this->t('Maximum number of entities to process per rule during the automatic post-sanitize run.'),
    ];

    // Available strategies (treat items as objects, not arrays).
    $rows = [];
    foreach ($this->strategyManager->all() as $id => $strategy) {
      $label = method_exists($strategy, 'label') ? (string) $strategy->label() : (string) $id;
      $rows[] = $id . ' — ' . $label;
    }
    sort($rows, SORT_NATURAL | SORT_FLAG_CASE);

    $form['available_strategies'] = [
      '#type' => 'item',
      '#title' => $this->t('Available strategies'),
      '#markup' => '<pre>' . implode("\n", array_map('htmlspecialchars', $rows)) . '</pre>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Parse rules from textarea into structured array.
    $lines = preg_split('/\R+/', (string) $form_state->getValue('replacements')) ?: [];
    $rules = [];

    $valid = array_keys($this->strategyManager->all());

    foreach ($lines as $i => $line) {
      $line = trim($line);
      if ($line === '' || str_starts_with($line, '#')) {
        continue;
      }
      if (!preg_match('/^([a-z0-9_]+)\.([a-z0-9_]+)\.([a-z0-9_]+)\s*=\s*([a-z0-9_]+)$/', $line, $m)) {
        $form_state->setErrorByName('replacements', $this->t('Invalid rule on line @n: %line', [
          '@n' => $i + 1,
          '%line' => $line,
        ]));
        continue;
      }
      $strategy = $m[4];
      if (!in_array($strategy, $valid, TRUE)) {
        $form_state->setErrorByName('replacements', $this->t('Unknown strategy "@s" on line @n. Known: @k', [
          '@s' => $strategy,
          '@n' => $i + 1,
          '@k' => implode(', ', $valid),
        ]));
        continue;
      }

      $rules[] = [
        'entity' => $m[1],
        'bundle' => $m[2],
        'field' => $m[3],
        'strategy' => $strategy,
      ];
    }

    // Store parsed rules into form state so we can save them without submit().
    $form_state->setValue('replacements_parsed', $rules);

    // Only persist if there are no validation errors.
    if (empty($form_state->getErrors())) {
      $this->configFactory->getEditable('sanitize_placeholder.settings')
        ->set('username_max_length', (int) $form_state->getValue('username_max_length'))
        ->set('faker_locale', (string) $form_state->getValue('faker_locale'))
        ->set('deterministic', (bool) $form_state->getValue('deterministic'))
        ->set('replacements', $form_state->getValue('replacements_parsed') ?? [])
        ->set('post_hook_limit', (int) $form_state->getValue('post_hook_limit'))
        ->save();
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * Turn stored replacements into textarea lines.
   */
  private function stringifyRules(array $rules): string {
    $out = [];
    foreach ($rules as $r) {
      $out[] = sprintf(
        '%s.%s.%s=%s',
        $r['entity'] ?? '',
        $r['bundle'] ?? '',
        $r['field'] ?? '',
        $r['strategy'] ?? ''
      );
    }
    return implode("\n", array_filter($out));
  }

}
