<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder\Service;

use Drupal\Component\Utility\Unicode;
use Drupal\Core\Config\ConfigFactoryInterface;
use Faker\Factory;
use Psr\Log\LoggerInterface;

/**
 * Faker wrapper with algorithmic (no-list) fallbacks.
 *
 * - Respects the 'faker_locale' config if set.
 * - Provides seed() for deterministic runs (when Faker is available).
 * - Exposes generator(), get(), and faker() (alias) to maintain
 *   compatibility with strategies that expect any of these methods.
 * - Falls back to simple algorithmic generators when Faker isn't installed.
 */
final class ThematicFaker {

  /**
   * Core Faker generator (NULL when Faker is not installed / failed to load).
   *
   * @var \Faker\Generator|null
   */
  private $faker = NULL;

  /**
   * Alphabet used to build pseudo-words (vowels).
   *
   * @var string
   */
  private const VOWELS = 'aeiou';

  /**
   * Alphabet used to build pseudo-words (consonants).
   *
   * @var string
   */
  private const CONS = 'bcdfghjklmnpqrstvwxyz';

  /**
   * Construct the wrapper.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory to read sanitize_placeholder.settings.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel (e.g. @logger.channel.sanitize_placeholder).
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
  ) {
    // Only attempt to create Faker if the library is installed.
    try {
      if (\class_exists(Factory::class)) {
        $cfg = $this->configFactory->get('sanitize_placeholder.settings');
        $locale = (string) ($cfg->get('faker_locale') ?? '');
        $this->faker = Factory::create($locale !== '' ? $locale : 'en_US');
      }
    }
    catch (\Throwable $e) {
      // If Faker cannot be created, quietly fall back to NULL and log.
      $this->faker = NULL;
      $this->logger->notice(
        'Faker not used (fallback active): @class - @message',
        [
          '@class' => \get_class($e),
          '@message' => $e->getMessage(),
          'exception' => $e,
        ]
      );
    }
  }

  /**
   * Whether a Faker generator is available.
   *
   * @return bool
   *   TRUE when Faker is present and usable, FALSE otherwise.
   */
  public function hasFaker(): bool {
    return $this->faker !== NULL;
  }

  /**
   * Seed the Faker generator (deterministic output when a seed is provided).
   *
   * No-op when Faker is not installed.
   *
   * @param int|null $seed
   *   Seed value or NULL to keep current randomness.
   */
  public function seed(?int $seed): void {
    if ($seed !== NULL && $this->faker !== NULL) {
      $this->faker->seed($seed);
    }
  }

  /**
   * Return the underlying Faker generator, or NULL if unavailable.
   *
   * @return \Faker\Generator|null
   *   The Faker generator or NULL.
   */
  public function generator() {
    return $this->faker;
  }

  /**
   * Back-compat alias used by some older strategies.
   *
   * @return \Faker\Generator|null
   *   The Faker generator or NULL.
   */
  public function get() {
    return $this->faker;
  }

  /**
   * Back-compat alias for code that calls $this->faker->faker().
   *
   * @return \Faker\Generator|null
   *   The Faker generator or NULL.
   */
  public function faker() {
    return $this->faker;
  }

  /**
   * Generate a first name.
   *
   * Uses Faker's userName() when available, otherwise an algorithmic fallback.
   *
   * @param int $maxLength
   *   Maximum length to enforce.
   *
   * @return string
   *   Username not exceeding $maxLength.
   *
   * @throws \Exception
   */
  public function username(int $maxLength = 50): string {
    $u = '';

    if ($this->faker !== NULL) {
      $u = (string) $this->faker->userName();
    }
    else {
      $u = $this->generateHandleLike();
    }

    if (\function_exists('mb_strlen') && \function_exists('mb_substr')) {
      return mb_strlen($u, 'UTF-8') > $maxLength ? mb_substr($u, 0, $maxLength, 'UTF-8') : $u;
    }
    // Fallback (Drupal ships with mbstring, so this is mostly theoretical).
    return \mb_strlen($u) > $maxLength ? \mb_substr($u, 0, $maxLength) : $u;
  }

  /**
   * Convenience: first name from current locale provider or fallback.
   *
   * @return string
   *   A plausible first name.
   *
   * @throws \Exception
   */
  public function firstName(): string {
    if ($this->faker !== NULL) {
      return (string) $this->faker->firstName();
    }
    return $this->generateSyllableName(3, 8, TRUE);
  }

  /**
   * Convenience: last name from current locale provider or fallback.
   *
   * @return string
   *   A plausible last name.
   *
   * @throws \Exception
   */
  public function lastName(): string {
    if ($this->faker !== NULL) {
      return (string) $this->faker->lastName();
    }
    return $this->generateSyllableName(4, 10, TRUE);
  }

  /**
   * Generate a simple "handle-like" identifier.
   *
   * @return string
   *   A handle-like string.
   *
   * @throws \Exception
   */
  private function generateHandleLike(): string {
    $left = $this->generateSyllableName(4, 8, FALSE);
    $right = $this->generateSyllableName(3, 5, FALSE);
    $handle = $left . '_' . $right;

    // Lowercase safely when available.
    if (\function_exists('mb_strtolower')) {
      return mb_strtolower($handle, 'UTF-8');
    }
    return \mb_strtolower($handle);
  }

  /**
   * Generate a pronounceable-ish fake name from consonant-vowel syllables.
   *
   * @param int $minLen
   *   Minimum length.
   * @param int $maxLen
   *   Maximum length.
   * @param bool $ucFirst
   *   Whether to ucfirst() the result.
   *
   * @return string
   *   Generated fake name.
   *
   * @throws \Exception
   */
  private function generateSyllableName(int $minLen, int $maxLen, bool $ucFirst): string {
    $vowels = ['a', 'e', 'i', 'o', 'u'];
    $consonants = ['b', 'c', 'd', 'f', 'g', 'h', 'j', 'k', 'l', 'm', 'n', 'p', 'q', 'r', 's', 't', 'v', 'w', 'x', 'z'];

    $len = \max($minLen, \min($maxLen, 2 * \random_int(2, 6)));
    $out = '';
    $toggleVowel = (bool) \random_int(0, 1);

    while (\strlen($out) < $len) {
      if ($toggleVowel) {
        $out .= $vowels[\array_rand($vowels)];
      }
      else {
        $out .= $consonants[\array_rand($consonants)];
      }
      $toggleVowel = !$toggleVowel;
    }

    $out = \mb_substr($out, 0, $len);
    if ($ucFirst) {
      return Unicode::ucfirst($out);
    }
    return $out;
  }

}
