<?php

namespace Drupal\sanitize_placeholder\Service;

use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Helper service to generate normalized, unique usernames/user handles.
 */
final readonly class UsernameGenerator {

  /**
   * Constructor.
   *
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   The transliteration service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   Logger channel (e.g. "sanitize_placeholder").
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    private TransliterationInterface $transliteration,
    private LoggerChannelInterface $logger,
    private ConfigFactoryInterface $configFactory,
    private EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Generate a suitable value for a username-like field.
   *
   * Assembles a base handle from first/last name fields (with fallbacks),
   * transliterates to ASCII, normalizes, enforces max length, and ensures
   * uniqueness for the target field.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being updated.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $fieldDefinition
   *   The field definition for the destination field.
   *
   * @return string
   *   A normalized, unique username/handle.
   */
  public function generate(
    EntityInterface $entity,
    FieldDefinitionInterface $fieldDefinition,
  ): string {
    $base = $this->buildBase($entity);
    $normalized = $this->normalize($base);

    // Enforce max length from field setting and module config
    // (username_max_length).
    $max = $this->resolveMaxLength($fieldDefinition, $entity);
    $normalized = $this->limitLength($normalized, $max);

    // Ensure uniqueness inside this field for this entity type.
    return $this->ensureUnique($normalized, $entity, $fieldDefinition, $max);
  }

  /**
   * Build a base string from common name fields with fallbacks.
   */
  private function buildBase(EntityInterface $entity): string {
    $candidates = [];

    foreach (['field_user_first_name', 'field_first_name'] as $first) {
      if ($entity->hasField($first) && !$entity->get($first)->isEmpty()) {
        $candidates[] = (string) $entity->get($first)->value;
        break;
      }
    }
    foreach (['field_user_last_name', 'field_last_name'] as $last) {
      if ($entity->hasField($last) && !$entity->get($last)->isEmpty()) {
        $candidates[] = (string) $entity->get($last)->value;
        break;
      }
    }

    if (!$candidates) {
      $label = (string) $entity->label();
      if ($label !== '') {
        $candidates[] = $label;
      }
    }

    if (!$candidates) {
      $candidates[] = 'user';
      if (!$entity->isNew()) {
        $candidates[] = (string) $entity->id();
      }
    }

    return implode('.', $candidates);
  }

  /**
   * Transliterate and normalize to a safe handle: [a-z0-9._-].
   */
  private function normalize(string $value): string {
    $ascii = $this->transliteration->transliterate($value, 'en');
    $lower = mb_strtolower($ascii);

    $sanitized = preg_replace('/[^a-z0-9._-]+/u', '_', $lower) ?? '';
    $sanitized = preg_replace('/_+/', '_', $sanitized) ?? '';
    $sanitized = trim($sanitized, '._-');

    if ($sanitized === '') {
      $sanitized = 'user';
    }
    return $sanitized;
  }

  /**
   * Determine the maximum length to respect.
   *
   * Chooses the minimum of:
   * - Field storage setting "max_length" (if present and > 0)
   * - Module config "sanitize_placeholder.settings:username_max_length"
   *   (if > 0)
   */
  private function resolveMaxLength(
    FieldDefinitionInterface $fieldDefinition,
    EntityInterface $entity,
  ): int {
    $limits = [];

    $fieldMax = $fieldDefinition->getSetting('max_length');
    if (is_numeric($fieldMax) && (int) $fieldMax > 0) {
      $limits[] = (int) $fieldMax;
    }

    $isUserName =
      $entity->getEntityTypeId() === 'user'
      && $fieldDefinition->getName() === 'name';

    $cfgMax = $this->configFactory->get('sanitize_placeholder.settings')->get('username_max_length');
    if (is_numeric($cfgMax) && (int) $cfgMax > 0 && $isUserName) {
      $limits[] = (int) $cfgMax;
    }

    if (!$limits) {
      return 60;
    }
    return max(1, min($limits));
  }

  /**
   * The mb_* safe length limiter.
   */
  private function limitLength(string $value, int $max): string {
    if ($max <= 0) {
      return $value;
    }
    return (mb_strlen($value, 'UTF-8') > $max)
      ? mb_substr($value, 0, $max, 'UTF-8')
      : $value;
  }

  /**
   * Ensure uniqueness for the destination field on the entity’s type.
   *
   * Appends a numeric suffix (_2, _3, …) if the candidate already exists,
   * honoring the maximum length constraint.
   */
  private function ensureUnique(string $candidate, EntityInterface $entity, FieldDefinitionInterface $fieldDefinition, int $maxLength): string {
    $entityTypeId = $entity->getEntityTypeId();
    $fieldName = $fieldDefinition->getName();

    // Try to obtain storage; if it fails, return a length-limited candidate.
    try {
      $storage = $this->entityTypeManager->getStorage($entityTypeId);
    }
    catch (\Throwable $e) {
      $this->logger->warning(
        'Failed to get storage for @type while ensuring uniqueness: @message',
        [
          '@type' => $entityTypeId,
          '@message' => $e->getMessage(),
          'exception' => $e,
        ]
      );
      return $this->limitLength($candidate, $maxLength);
    }

    $exists = static function (string $value) use ($storage, $entity, $fieldName): bool {
      $query = $storage->getQuery()->accessCheck(FALSE);
      $query->condition($fieldName, $value);

      if (!$entity->isNew()) {
        $idKey = $entity->getEntityType()->getKey('id');
        if ($idKey) {
          $query->condition($idKey, $entity->id(), '<>');
        }
      }

      $query->range(0, 1);
      $result = $query->execute();
      return !empty($result);
    };

    if (!$exists($candidate)) {
      return $candidate;
    }

    $base = $candidate;
    $i = 2;
    while (TRUE) {
      $suffix = '_' . $i;
      $trimTo = max(1, $maxLength - mb_strlen($suffix, 'UTF-8'));
      $try = $this->limitLength($base, $trimTo) . $suffix;

      if (!$exists($try)) {
        return $try;
      }
      $i++;

      if ($i > 1000) {
        $this->logger->warning('Could not find a unique username candidate after 1000 attempts. Using last tried value.');
        return $try;
      }
    }
  }

}
