<?php

namespace Drupal\sanitize_placeholder\Strategy;

/**
 * Collects and resolves placeholder strategies.
 */
final class StrategyManager {

  /**
   * Strategy Manager.
   *
   * @var array<string,\Drupal\sanitize_placeholder\Strategy\StrategyInterface>
   *   Map of strategy id => strategy service.
   */
  private array $strategies = [];

  /**
   * Services tagged.
   *
   * @param iterable<\Drupal\sanitize_placeholder\Strategy\StrategyInterface> $strategies
   *   All services tagged with "sanitize_placeholder.strategy".
   */
  public function __construct(iterable $strategies) {
    foreach ($strategies as $service) {
      if (!$service instanceof StrategyInterface) {
        // Ignore anything not implementing our contract.
        continue;
      }

      // Primary id from the strategy itself (if provided).
      $ids = [];
      if (method_exists($service, 'id')) {
        $id = trim((string) $service->id());
        if ($id !== '') {
          $ids[] = $id;
        }
      }

      // Fallback id from class name, e.g. FirstNameStrategy => first_name.
      $class = (new \ReflectionClass($service))->getShortName();
      if (str_ends_with($class, 'Strategy')) {
        $class = mb_substr($class, 0, -8);
      }
      $ids[] = $this->toSnakeCase($class);

      // Register all discovered ids (first one wins).
      foreach ($ids as $key) {
        $key = mb_strtolower($key);
        if (!isset($this->strategies[$key])) {
          $this->strategies[$key] = $service;
        }
      }
    }
  }

  /**
   * Return all strategies keyed by normalized id.
   *
   * @return array<string,\Drupal\sanitize_placeholder\Strategy\StrategyInterface>
   *   Return all strategies.
   */
  public function all(): array {
    return $this->strategies;
  }

  /**
   * Get a strategy by id.
   *
   * @param string $id
   *   Strategy id (e.g., 'first_name', 'username').
   *
   * @return \Drupal\sanitize_placeholder\Strategy\StrategyInterface
   *   Return strategy interface by id.
   */
  public function get(string $id): StrategyInterface {
    $key = mb_strtolower($id);
    if (!isset($this->strategies[$key])) {
      throw new \InvalidArgumentException(sprintf('Unknown strategy "%s".', $id));
    }
    return $this->strategies[$key];
  }

  /**
   * Convert "CamelCase" or "mixed-Case" to "snake_case".
   */
  private function toSnakeCase(string $name): string {
    $name = preg_replace('/([a-z])([A-Z])/u', '$1_$2', $name);
    $name = preg_replace('/[^A-Za-z0-9]+/u', '_', $name);
    return mb_strtolower(trim((string) $name, '_'));
  }

}
