<?php

namespace Drupal\sanitize_placeholder\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\sanitize_placeholder\Service\UsernameGenerator;

/**
 * Generates usernames using the UsernameGenerator, then normalizes separators.
 *
 * Normalization rules:
 *  - Replace '.' with '_'.
 *  - Replace spaces with '_'.
 *  - Collapse multiple underscores.
 *  - Trim leading/trailing underscores.
 */
final class UsernameStrategy implements StrategyInterface {

  /**
   * The username generator service.
   *
   * @var \Drupal\sanitize_placeholder\Service\UsernameGenerator
   */
  private UsernameGenerator $generator;

  /**
   * Constructs a UsernameStrategy object.
   *
   * @param \Drupal\sanitize_placeholder\Service\UsernameGenerator $generator
   *   The username generator service.
   */
  public function __construct(UsernameGenerator $generator) {
    $this->generator = $generator;
  }

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'username';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return 'Username';
  }

  /**
   * {@inheritdoc}
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $field_definition): string {
    // Let the service produce the base username.
    $username = $this->generator->generate($entity, $field_definition);
    $username = str_replace('.', '_', $username);
    $username = preg_replace('/\s+/', '_', $username) ?? $username;
    $username = preg_replace('/_+/', '_', $username) ?? $username;
    return trim($username, '_');
  }

}
