<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api_example\Plugin\ScannerFixer\Fixer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\node\NodeInterface;
use Drupal\scanner_fixer_api\Annotation\Fixer;
use Drupal\scanner_fixer_api\Fixer\FixerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Re-insert a node into the taxonomy index.
 *
 * A Fixer is a way to act upon the things found by a Scanner, as the second
 * step in a Solution.
 *
 * Fixers aren't designed to be run on their own; rather, they're intended to be
 * run as part of a Solution. That being said, there are Drush commands (that
 * are intended to assist with development) to:
 *
 * 1. list Fixers (`scanner-fixer:list-fixers`), and,
 * 2. run an individual Fixer on a single ID (`scanner-fixer:run-fixer`).
 *
 * Note that Fixers should only be used with other Fixers and Scanners that all
 * agree on the format of the item IDs returned by the Scanner(s). For example,
 * if you combine a Scanner that returns NIDs with a Fixer that assumes it is
 * ingesting TIDs, you will have problems.
 *
 * The TaxonomyIndexNode Fixer below uses dependency injection to get the Node
 * storage handler, which it uses to load Node IDs passed to it by Scanners. It
 * calls \taxonomy_build_node_index() on the loaded node.
 */
#[Fixer(
  id: "taxonomy_index_node",
  title: new TranslatableMarkup("Taxonomy-index a node"),
)]
class TaxonomyIndexNode implements FixerInterface, ContainerFactoryPluginInterface {

  /**
   * A configuration array containing information about the plugin instance.
   *
   * @var array
   */
  protected array $configuration;

  /**
   * An entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The plugin implementation definition.
   *
   * @var mixed
   */
  protected mixed $pluginDefinition;

  /**
   * The plugin ID for the plugin instance.
   *
   * @var string
   */
  protected string $pluginId;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static();
    $instance->configuration = $configuration;
    $instance->pluginId = $plugin_id;
    $instance->pluginDefinition = $plugin_definition;
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function canFix(mixed $item): bool {
    // We can act on this item if it is a valid node ID.
    return (\is_string($item) || \is_int($item))
      && !\is_null($this->loadNodeWithId($item));
  }

  /**
   * {@inheritdoc}
   */
  public function performFix(mixed $item): bool {
    try {
      \taxonomy_build_node_index($this->loadNodeWithId($item));
    }
    // If we catch an exception, return that the operation was a failure.
    catch (\Throwable) {
      return FALSE;
    }

    // If we get here, we were successful.
    return TRUE;
  }

  /**
   * Load a node with a given ID.
   *
   * @param int|string $nid
   *   The ID of the node to load.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The loaded node, or NULL if a node with the given ID cannot be found.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   Thrown if the storage handler couldn't be loaded.
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   Thrown if the entity type doesn't exist.
   */
  protected function loadNodeWithId(int|string $nid): ?NodeInterface {
    return $this->entityTypeManager->getStorage('node')->load($nid);
  }

}
