<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Drush\Commands;

use Consolidation\AnnotatedCommand\Attributes\Command;
use Consolidation\AnnotatedCommand\Attributes\Help;
use Consolidation\AnnotatedCommand\Attributes\Usage;
use Consolidation\OutputFormatters\StructuredData\ListDataInterface;
use Consolidation\OutputFormatters\StructuredData\UnstructuredListData;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\scanner_fixer_api\Scanner\ScannerPluginManager;
use Drush\Attributes\Argument;
use Drush\Attributes\Bootstrap;
use Drush\Boot\DrupalBootLevels;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands related to Scanner-Fixer Scanners.
 */
final class ScannerCommands extends DrushCommands implements ContainerInjectionInterface {

  /**
   * Plugin manager for scanners.
   *
   * @var \Drupal\scanner_fixer_api\Scanner\ScannerPluginManager
   */
  protected ScannerPluginManager $scannerPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->scannerPluginManager = $container->get('plugin.manager.scanner_fixer_api.scanner');
    return $instance;
  }

  /**
   * List available scanners.
   *
   * @param array $options
   *   An associative array of command options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\ListDataInterface
   *   A list of available scanners.
   */
  #[Command(name: 'scanner-fixer:list-scanners')]
  #[Help(description: 'Lists available scanners.')]
  #[Usage(name: 'drush scanner-fixer:list-scanners', description: 'List available scanners.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function listScanners(array $options = ['format' => 'json']): ListDataInterface {
    $scannerDefinitions = $this->scannerPluginManager->getDefinitions();
    return new UnstructuredListData($scannerDefinitions);
  }

  /**
   * Run an individual scanner.
   *
   * @param string $scannerId
   *   The ID of the scanner to run.
   * @param array $options
   *   An associative array of command options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\ListDataInterface
   *   A list of data found by the scanner.
   */
  #[Command(name: 'scanner-fixer:run-scanner')]
  #[Argument(name: 'scannerId', description: 'The ID of the scanner to run')]
  #[Help(description: 'Runs a scanner-fixer scanner.')]
  #[Usage(name: 'drush scanner-fixer:run-scanner scannerId', description: 'Run the scanner named scannerId.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function runScanner(string $scannerId, array $options = ['format' => 'json']): ListDataInterface {
    /** @var \Drupal\scanner_fixer_api\Scanner\ScannerInterface $scanner */
    $scanner = $this->scannerPluginManager->createInstance($scannerId);
    $items = $scanner->findProblemIds();
    return new UnstructuredListData($items);
  }

}
