<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Form;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\scanner_fixer_api\Model\SolutionStatCounter;
use Drupal\scanner_fixer_api\Solution\SolutionPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * A multi-step form to run a solution.
 */
class SolutionWizard extends FormBase {

  /**
   * Plugin manager for solutions.
   *
   * @var \Drupal\scanner_fixer_api\Solution\SolutionPluginManager
   */
  protected SolutionPluginManager $solutionPluginManager;

  /**
   * Display fix operation results.
   *
   * Callback for \batch_set().
   *
   * @param bool $success
   *   A boolean indicating whether the batch has completed successfully.
   * @param \Drupal\scanner_fixer_api\Model\SolutionStatCounter $results
   *   The value set in $context['results'] by callback_batch_operation().
   * @param array $operations
   *   If $success is FALSE, contains the operations that remained unprocessed.
   * @param string|\Stringable $elapsed
   *   A string representing the elapsed time for the batch process, e.g.,
   *   '1 min 30 secs'.
   *
   * @see \callback_batch_finished()
   */
  public static function batchFinished(bool $success, SolutionStatCounter $results, array $operations, string|\Stringable $elapsed): void {
    if ($success) {
      $message = new PluralTranslatableMarkup(
        $results->getNumberTried(),
        'Processed one item.',
        'Processed @count items.',
      );
    }
    else {
      $message = new TranslatableMarkup('Finished with an error.');
    }
    \Drupal::messenger()->addStatus($message);
    \Drupal::messenger()->addStatus(new TranslatableMarkup('Detailed results: @results.', [
      '@results' => $results->getSummaryString(),
    ]));
  }

  /**
   * Perform a fix on a single item.
   *
   * @param mixed $itemId
   *   The ID of the item to fix.
   * @param string $solutionId
   *   The ID of the solution to run.
   * @param array|\ArrayAccess $context
   *   The batch context array, passed by reference.
   *
   * @see \callback_batch_operation()
   */
  public static function batchOperationCallbackFix(mixed $itemId, string $solutionId, array|\ArrayAccess $context): void {
    // If there are no results, initialize a new stats counter.
    if (empty($context['results'])) {
      $context['results'] = new SolutionStatCounter();
    }

    // Load the solution and run the fixers on the item we were passed.
    try {
      /** @var \Drupal\scanner_fixer_api\Solution\SolutionInterface $solution */
      $solution = \Drupal::service('plugin.manager.scanner_fixer_api.solution')
        ->createInstance($solutionId);

      // Run fixers on the current item, and add the results to the previous
      // results.
      $context['results']->add($solution->runFixers([$itemId]));
    }
    // At this point, if we can't find the solution, then do nothing so we skip
    // this item.
    catch (PluginNotFoundException | PluginException) {
      // No-op.
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->solutionPluginManager = $container->get('plugin.manager.scanner_fixer_api.solution');
    return $instance;
  }

  /**
   * Determine if the current user should have access to this solution wizard.
   *
   * @param string $solutionId
   *   The ID of the solution we are determining access to.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The result of the access check.
   *
   * @see \Drupal\scanner_fixer_api\Solution\SolutionPermissions::solutionsPermissions()
   */
  public function access(string $solutionId): AccessResult {
    return AccessResult::allowedIfHasPermission($this->currentUser(), "use solution $solutionId");
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $solutionId = '') {
    // Store the Solution ID.
    if (!empty($solutionId)) {
      $form_state->setValue('solution_id', $solutionId);
    }
    $form['solution_id'] = [
      '#type' => 'value',
      '#value' => $form_state->getValue('solution_id'),
    ];

    // Store the items to fix.
    $listOfItemsToFix = $form_state->getValue('items_to_fix') ?? [];
    $form['items_to_fix'] = [
      '#type' => 'value',
      '#value' => $listOfItemsToFix,
    ];

    try {
      // Try to load the solution definition so we can display a 404 if it does
      // not exist.
      $this->solutionPluginManager->getDefinition($solutionId);

      // If the list of items to fix is empty, display the first page of the
      // form.
      if (empty($listOfItemsToFix)) {
        $form['list_items_to_fix'] = [
          '#type' => 'item',
          '#input' => FALSE,
          '#title' => t('No items to fix (yet)'),
          '#description' => t('Try clicking the "Search" button below.'),
        ];
        $form['actions']['search'] = [
          '#type' => 'submit',
          '#value' => t('Search for items to fix'),
          '#submit' => ['::submitPage1Form'],
        ];
      }
      // If there are items to fix, display the second page of the form.
      else {
        $numberOfItemsToFix = \count($listOfItemsToFix);
        $form['list_items_to_fix'] = [
          '#type' => 'item',
          '#input' => FALSE,
          '#title' => $this->formatPlural(
            $numberOfItemsToFix,
            'Found 1 item to fix',
            'Found @count items to fix',
          ),
          '#description' => $this->formatPlural(
            $numberOfItemsToFix,
            'ID of item to fix: %item_ids',
            'IDs of items to fix: %item_ids',
            ['%item_ids' => \implode(', ', $listOfItemsToFix)],
          ),
        ];
        $form['actions']['fix'] = [
          '#type' => 'submit',
          '#value' => $this->formatPlural(
            $numberOfItemsToFix,
            'Fix listed item',
            'Fix listed items',
          ),
        ];
      }
    }
    // If the solution plugin manager reports an error, return a 404.
    catch (PluginNotFoundException) {
      throw new NotFoundHttpException();
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'scanner_fixer_api_solution_wizard';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get the solution ID and list of items to fix from the form state.
    $solutionId = $form_state->getValue('solution_id');
    $listOfItemsToFix = $form_state->getValue('items_to_fix') ?? [];

    // Progressive batches are backed by the database. Add a meaningful title
    // and a custom finish callback.
    $batchBuilder = (new BatchBuilder())
      ->setProgressive()
      ->setTitle(t('Scanner-fixer'))
      ->setFinishCallback('\Drupal\scanner_fixer_api\Form\SolutionWizard::batchFinished');

    // Add each item as a separate operation.
    foreach ($listOfItemsToFix as $itemId) {
      $batchBuilder->addOperation(
        '\Drupal\scanner_fixer_api\Form\SolutionWizard::batchOperationCallbackFix',
        [$itemId, $solutionId],
      );
    }

    // Set the batch.
    \batch_set($batchBuilder->toArray());
  }

  /**
   * Form submission handler for the first page of the form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitPage1Form(array &$form, FormStateInterface $form_state): void {
    $solutionId = $form_state->getValue('solution_id');

    try {
      // Initialize the solution.
      /** @var \Drupal\scanner_fixer_api\Solution\SolutionInterface $solution */
      $solution = $this->solutionPluginManager->createInstance($solutionId);

      // Scan for the items we need and store them to the form state.
      $itemsToFix = $solution->runScanners();
      $form_state->setValue('items_to_fix', $itemsToFix);
    }
    // If the solution plugin manager reports an error, return a 404.
    catch (PluginNotFoundException | PluginException) {
      throw new NotFoundHttpException();
    }

    // Tell the form state to reload.
    $form_state->setRebuild();
  }

  /**
   * Get the page title for the Solution wizard form.
   *
   * @param string|null $solutionId
   *   The ID of the solution, from the URL.
   *
   * @return string|\Stringable
   *   A title for the Solution wizard form.
   */
  public function title(?string $solutionId = ''): string|\Stringable {
    $title = '';

    // Try loading the solution definition and getting the solution's title.
    try {
      if ($solutionDefinition = $this->solutionPluginManager->getDefinition($solutionId)) {
        $title = new FormattableMarkup('@title', [
          '@title' => $solutionDefinition['title'],
        ]);
      }
    }
    // If a plugin cannot be found, do nothing, i.e.: leave $title set to an
    // empty string.
    catch (PluginNotFoundException) {
      // No-op.
    }

    return $title;
  }

}
