<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api_example\Plugin\ScannerFixer\Solution;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\scanner_fixer_api\Annotation\Solution;
use Drupal\scanner_fixer_api\Solution\SolutionBase;
use Drupal\scanner_fixer_api_example\Plugin\ScannerFixer\Fixer\TaxonomyIndexNode;
use Drupal\scanner_fixer_api_example\Plugin\ScannerFixer\Scanner\PublishedNodeScanner;

/**
 * Rebuild the taxonomy_index table by re-indexing all published nodes.
 *
 * A Solution is a way to group Scanners and Fixers.
 *
 * Solutions are designed to be the things that you normally run with the
 * Scanner-Fixer API. There is:
 *
 * 1. a user interface to list and run Solutions at
 *    `/admin/content/scanner_fixer_api`,
 * 2. a Drush command to list Solutions (`scanner-fixer:list-solutions`), and,
 * 3. a Drush command to run an individual Solution
 *     (`scanner-fixer:run-solution`).
 *
 * Solutions are designed to be easily composed - in code - from one or more
 * Scanners, and one or more Fixers, i.e.: using a Solution annotation like the
 * one below.
 *
 * Note that *all* of the Scanners and Fixers that make up a Solution *must*
 * agree on an item ID format. That is to say, the item IDs produced by the
 * Scanner(s) and acted upon by the Fixer(s) must either all refer to the same
 * thing (e.g.: all be node IDs), or must be unambiguous in some way (e.g.:
 * 'entity_type/NID' or a UUID). Put another way, creating a Solution that
 * combines a Scanner that returns Node IDs, with a Fixer that loads and acts on
 * Taxonomy Term IDs will produce undesirable results!
 *
 * Solutions that extend SolutionBase probably won't need to implement any
 * functions; but it is possible to write stand-alone Solutions by directly
 * implementing SolutionInterface.
 *
 * Note that Drupal's convention is to refer to plugins (like Scanners, Fixers,
 * and Solutions) by their Plugin ID, but in this case, to make it easier to
 * navigate using a development-editor, we reference Scanners and Fixers by
 * their class names.
 *
 * The RebuildTaxonomyIndex Solution below finds published nodes to operate on
 * using the example PublishedNodeScanner scanner (which produces a list of node
 * IDs); then re-indexes them using the example TaxonomyIndexNode fixer (which
 * loads the nodes with the given IDs and passes them to
 * \taxonomy_build_node_index() for reindexing).
 */
#[Solution(
  id: "scanner_fixer_api_example_reindex_nodes",
  title: new TranslatableMarkup("Rebuild taxonomy index"),
  description: new TranslatableMarkup("Rebuilds the taxonomy_index table by rebuilding all nodes."),
  scanners: [PublishedNodeScanner::class],
  fixers: [TaxonomyIndexNode::class],
)]
class RebuildTaxonomyIndex extends SolutionBase {}
