<?php

declare(strict_types=1);

namespace Drupal\Tests\scanner_fixer_api\Functional;

use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Solution overview page.
 *
 * Note this test installs the scanner_fixer_api_test test module which defines
 * a handful of solutions. Contrast this with the SolutionOverviewEmptyTest test
 * which only installs the base scanner_fixer_api module, which doesn't define
 * any solutions.
 *
 * @group scanner_fixer_api
 */
class SolutionOverviewTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['scanner_fixer_api_test'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Test that you can only view the overview page if you have permission.
   */
  public function testNeedPermissionToViewSolutions(): void {
    // Setup: Create a user without the overview page permission and log in.
    $this->drupalLogin($this->createUser([]));

    // SUT: Load the Solution overview page.
    $this->drupalGet('/admin/content/scanner_fixer_api');

    // Assert: We should get an Access Denied.
    $this->assertSession()->statusCodeEquals(403);

    // Setup: Create a user with the overview page permission and log in.
    $this->drupalLogin($this->createUser([
      'use scanner-fixer solution overview page',
    ]));

    // SUT: Load the Solution overview page.
    $this->drupalGet('/admin/content/scanner_fixer_api');

    // Assert: We should get a success error code and see the title of the page.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Scanner-fixer solution list');

    // Assert: We should see the solution titles and descriptions defined in the
    // test module; and the solution titles should be linked to the wizard.
    $this->assertSession()->pageTextContains('Test solution: Five items scanner');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'five_items_scanner',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds five items to fix.');

    $this->assertSession()->pageTextContains('Test solution: Incapable fixer');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'incapable_fixer',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds one item and whose fixer says it cannot fix the item.');

    $this->assertSession()->pageTextContains('Test solution: Two scanners');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'two_scanners_solution',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer that uses two scanners.');

    $this->assertSession()->pageTextContains('Test solution: One item scanner');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'one_items_scanner',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds one item to fix.');

    $this->assertSession()->pageTextContains('Test solution: Capable, failure fixer');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'capable_failure_fixer',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds one item and whose fixer says it can fix an item, but fails to follow through with that.');

    $this->assertSession()->pageTextContains('Test solution: Capable, successful fixer');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'capable_successful_fixer',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds one item and whose fixer says it can fix an item, then does so successfully.');

    $this->assertSession()->pageTextContains('Test solution: Two capable, successful fixers');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'two_capable_successful_fixers',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer that uses two instances of the capable, successful fixer.');

    $this->assertSession()->pageTextContains('Test solution: Exception fixer');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'exception_fixer',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds one item and whose fixer throws an exception when trying to fix the item.');

    $this->assertSession()->pageTextContains('Test solution: Zero items scanner');
    $this->assertSession()->linkByHrefExists(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => 'zero_items_scanner',
    ])->toString());
    $this->assertSession()->pageTextContains('A scanner-fixer whose scanner finds nothing to fix.');
  }

}
