<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api_example\Plugin\ScannerFixer\Scanner;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\scanner_fixer_api\Annotation\Scanner;
use Drupal\scanner_fixer_api\Scanner\ScannerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Find published nodes.
 *
 * A Scanner is a way to find things to fix, as the first step in a Solution.
 *
 * Scanners aren't designed to be run on their own; rather, they're intended to
 * be run as part of a Solution. That being said, there are Drush commands (that
 * are intended to assist with development) to:
 *
 * 1. list Scanners (`scanner-fixer:list-scanners`), and,
 * 2. run an individual Scanner (`scanner-fixer:run-scanner`).
 *
 * Note that Scanners should only be used with other Scanners and Fixers that
 * all agree on the format of the item IDs returned by findProblemIds(). For
 * example, if you combine a Scanner that returns NIDs with a Scanner that
 * returns TIDs; or if you combine a Scanner that returns NIDs with a Fixer that
 * assumes it is ingesting TIDs, you will have problems.
 *
 * The PublishedNodeScanner Scanner below uses dependency injection to get the
 * Node storage handler, which it uses to build an Entity Query that searches
 * for published nodes, and returns their node IDs.
 */
#[Scanner(
  id: "scanner_fixer_api_example_published_nodes",
  title: new TranslatableMarkup("Example: Published node scanner"),
)]
class PublishedNodeScanner extends ScannerBase {

  /**
   * An entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function findProblemIds(): array {
    return $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->execute();
  }

}
