<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Drush\Commands;

use Consolidation\AnnotatedCommand\Attributes\Command;
use Consolidation\AnnotatedCommand\Attributes\Help;
use Consolidation\AnnotatedCommand\Attributes\Usage;
use Consolidation\OutputFormatters\StructuredData\ListDataInterface;
use Consolidation\OutputFormatters\StructuredData\UnstructuredListData;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\scanner_fixer_api\Fixer\FixerPluginManager;
use Drush\Attributes\Argument;
use Drush\Attributes\Bootstrap;
use Drush\Boot\DrupalBootLevels;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands related to Scanner-Fixer Fixers.
 */
final class FixerCommands extends DrushCommands implements ContainerInjectionInterface {

  /**
   * Plugin manager for fixers.
   *
   * @var \Drupal\scanner_fixer_api\Fixer\FixerPluginManager
   */
  protected FixerPluginManager $fixerPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->fixerPluginManager = $container->get('plugin.manager.scanner_fixer_api.fixer');
    return $instance;
  }

  /**
   * List available fixers.
   *
   * @param array $options
   *   An associative array of command options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\ListDataInterface
   *   A list of available scanners.
   */
  #[Command(name: 'scanner-fixer:list-fixers')]
  #[Help(description: 'Lists available fixers.')]
  #[Usage(name: 'drush scanner-fixer:list-fixers', description: 'List available fixers.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function listFixers(array $options = ['format' => 'json']): ListDataInterface {
    $fixerDefinitions = $this->fixerPluginManager->getDefinitions();
    return new UnstructuredListData($fixerDefinitions);
  }

  /**
   * Run an individual fixer on a single item.
   *
   * @param string $fixerId
   *   The ID of the fixer to run.
   * @param string $itemId
   *   The ID of the item to run the fixer on.
   * @param array $options
   *   An associative array of command options.
   */
  #[Command(name: 'scanner-fixer:run-fixer')]
  #[Argument(name: 'fixerId', description: 'The ID of the fixer to run')]
  #[Argument(name: 'itemId', description: 'The ID of the item to run the fixer on')]
  #[Help(description: 'Runs a scanner-fixer fixer on a single item.')]
  #[Usage(name: 'drush scanner-fixer:run-fixer fixerId item_id', description: 'Run the fixer named fixerId.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function runFixer(string $fixerId, string $itemId, array $options = ['format' => 'json']): void {
    /** @var \Drupal\scanner_fixer_api\Fixer\FixerInterface $fixer */
    $fixer = $this->fixerPluginManager->createInstance($fixerId);
    if ($fixer->canFix($itemId)) {
      if ($fixer->performFix($itemId)) {
        $this->io()->note('The fix was successful.');
      }
      else {
        $this->io()->error('The fix was attempted but failed.');
      }
    }
    else {
      $this->io()->warning('The fixer said that it could not fix the given item.');
    }
  }

}
