<?php

declare(strict_types=1);

namespace Drupal\Tests\scanner_fixer_api\Functional;

use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Test the Solution wizard.
 *
 * @group scanner_fixer_api
 */
class SolutionWizardTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['scanner_fixer_api_test'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A data provider for testSolutionWizard().
   *
   * @return array[]
   *   An array of argument-arrays to pass to a test function. Each array should
   *   contain:
   *   1. The ID of the solution plugin to test.
   */
  public static function dataProviderSolutions(): array {
    return [
      'capable_failure_fixer' => [
        'capable_failure_fixer',
        'Test solution: Capable, failure fixer',
        1,
        'Processed one item.',
        'Detailed results: 0 successes, 1 failure, 0 skipped out of 1 total.',
      ],
      'capable_successful_fixer' => [
        'capable_successful_fixer',
        'Test solution: Capable, successful fixer',
        1,
        'Processed one item.',
        'Detailed results: 1 success, 0 failures, 0 skipped out of 1 total.',
      ],
      'exception_fixer' => [
        'exception_fixer',
        'Test solution: Exception fixer',
        1,
        'Processed one item.',
        'Detailed results: 0 successes, 1 failure, 0 skipped out of 1 total.',
      ],
      'five_items_scanner' => [
        'five_items_scanner',
        'Test solution: Five items scanner',
        5,
        'Processed 0 items.',
        'Detailed results: 0 successes, 0 failures, 0 skipped out of 0 total.',
      ],
      'incapable_fixer' => [
        'incapable_fixer',
        'Test solution: Incapable fixer',
        1,
        'Processed one item.',
        'Detailed results: 0 successes, 0 failures, 1 skipped out of 1 total.',
      ],
      'one_items_scanner' => [
        'one_items_scanner',
        'Test solution: One item scanner',
        1,
        'Processed 0 items.',
        'Detailed results: 0 successes, 0 failures, 0 skipped out of 0 total.',
      ],
      'two_capable_successful_fixers' => [
        'two_capable_successful_fixers',
        'Test solution: Two capable, successful fixers',
        1,
        'Processed one item.',
        'Detailed results: 1 success, 0 failures, 0 skipped out of 1 total.',
      ],
      'two_scanners_solution' => [
        'two_scanners_solution',
        'Test solution: Two scanners',
        6,
        'Processed 6 items.',
        'Detailed results: 6 successes, 0 failures, 0 skipped out of 6 total.',
      ],
      'zero_items_scanner' => [
        'zero_items_scanner',
        'Test solution: Zero items scanner',
        0,
        '',
        '',
      ],
    ];
  }

  /**
   * Test the wizards for each of the test solutions.
   *
   * @dataProvider dataProviderSolutions
   */
  public function testSolutionWizard(string $solutionId, string $solutionTitle, int $numItemsFound, string $processedMessage, string $detailedMessage): void {
    // Setup: Log in as an administrator that can access this solution wizard.
    $this->drupalLogin($this->createUser([
      "use solution $solutionId",
    ]));

    // SUT: Navigate to the wizard page.
    $this->drupalGet(Url::fromRoute('scanner_fixer_api.solution.wizard', [
      'solutionId' => $solutionId,
    ]));

    // Assert: Initially, there should be no items to fix, and a search button.
    $this->assertItemsToFix($solutionTitle, 0);

    // SUT: Run the search.
    $this->submitForm([], 'Search for items to fix');

    // If we expect more than 0 items to fix, then test fixing them.
    if ($numItemsFound > 0) {
      // Assert: We should see the expected number of items to fix.
      $this->assertItemsToFix($solutionTitle, $numItemsFound);

      // SUT: Run the fixers.
      $this->submitForm([], 'op');

      // Assert: The title of the page should always be the solution title.
      $this->assertSession()->pageTextContains($solutionTitle);

      // Assert: We should see the expected status messages.
      $this->assertSession()->statusMessageContains($processedMessage, 'status');
      $this->assertSession()->statusMessageContains($detailedMessage, 'status');
    }
    // Clicking the button again should keep us on step 1 of the form.
    else {
      $this->assertItemsToFix($solutionTitle, 0);
      $this->submitForm([], 'Search for items to fix');
      $this->assertItemsToFix($solutionTitle, 0);
      $this->assertSession()->buttonExists('Search for items to fix');
    }
  }

  /**
   * Check the form item on the first page to ensure it works the way we expect.
   *
   * @param string $solutionTitle
   *   The expected title of the page.
   * @param int $number
   *   The expected number of items we expect the solution to have found.
   */
  protected function assertItemsToFix(string $solutionTitle, int $number): void {
    // The title of the page should always be the solution title.
    $this->assertSession()->pageTextContains($solutionTitle);

    switch ($number) {
      case 0:
        $this->assertSession()->pageTextContains('No items to fix (yet)');
        $this->assertSession()->pageTextContains('Try clicking the "Search" button below.');
        $this->assertSession()->buttonExists('Search for items to fix');
        break;

      case 1:
        $this->assertSession()->pageTextContains('Found 1 item to fix');
        $this->assertSession()->pageTextContains('ID of item to fix:');
        $this->assertSession()->buttonExists('Fix listed item');
        break;

      default:
        $this->assertSession()->pageTextContains("Found $number items to fix");
        $this->assertSession()->pageTextContains('IDs of items to fix:');
        $this->assertSession()->buttonExists('Fix listed items');
        break;
    }
  }

}
