<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\scanner_fixer_api\Solution\SolutionPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays a list of available scanner-fixer solutions.
 */
class SolutionOverview extends ControllerBase {

  /**
   * Plugin manager for solutions.
   *
   * @var \Drupal\scanner_fixer_api\Solution\SolutionPluginManager
   */
  protected SolutionPluginManager $solutionPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->solutionPluginManager = $container->get('plugin.manager.scanner_fixer_api.solution');
    return $instance;
  }

  /**
   * Page callback: Displays a list of available scanner-fixer solutions.
   *
   * @return array
   *   A render array for a page containing a list of available scanner-fixer
   *   solutions.
   */
  public function listSolutions(): array {
    $build = [];

    // Prepare an output table with a row for each defined Solution.
    $rows = [];
    $header = [
      $this->t('Title'),
      $this->t('Description'),
    ];
    $solutions = $this->solutionPluginManager->getDefinitions();
    foreach ($solutions as $solutionId => $solutionInfo) {
      $rows[$solutionId] = [
        'title' => Link::fromTextAndUrl(
          $solutionInfo['title'],
          Url::fromRoute('scanner_fixer_api.solution.wizard', [
            'solutionId' => $solutionId,
          ]),
        ),
        'description' => $solutionInfo['description'],
      ];
    }
    $build['scanner_fixer_solution_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => t('No scanner-fixer solutions have been registered yet.'),
    ];

    return $build;
  }

}
