<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Model;

use Drupal\Core\StringTranslation\PluralTranslatableMarkup;

/**
 * A data object to represent the result of running fixers over a set of items.
 */
class SolutionStatCounter implements SolutionResultsInterface {

  /**
   * The number of items that were not fully fixed.
   *
   * @var int
   */
  private int $numFailure = 0;

  /**
   * The number of items that the fixer didn't think it could fix.
   *
   * @var int
   */
  private int $numSkipped = 0;

  /**
   * The number of items that were fully fixed.
   *
   * @var int
   */
  private int $numSuccess = 0;

  /**
   * The total number of items that were tried.
   *
   * @var int
   */
  private int $numTried = 0;

  /**
   * Add the results of another SolutionStatCounter to this one.
   *
   * @param self $other
   *   Another SolutionStatCounter to add to this one.
   *
   * @return self
   *   The updated SolutionStatCounter.
   */
  public function add(self $other): self {
    $this->numFailure += $other->getNumberOfFailures();
    $this->numSkipped += $other->getNumberOfSkipped();
    $this->numSuccess += $other->getNumberOfSuccesses();
    $this->numTried += $other->getNumberTried();

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getNumberOfFailures(): int {
    return $this->numFailure;
  }

  /**
   * {@inheritdoc}
   */
  public function getNumberOfSkipped(): int {
    return $this->numSkipped;
  }

  /**
   * {@inheritdoc}
   */
  public function getNumberOfSuccesses(): int {
    return $this->numSuccess;
  }

  /**
   * {@inheritdoc}
   */
  public function getNumberTried(): int {
    return $this->numTried;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummaryString(): string|\Stringable {
    $successMessage = new PluralTranslatableMarkup($this->numSuccess,
      '1 success',
      '@count successes',
    );

    $failureMessage = new PluralTranslatableMarkup($this->numFailure,
      '1 failure',
      '@count failures',
    );

    $skippedMessage = new PluralTranslatableMarkup($this->numSkipped,
      '1 skipped',
      '@count skipped',
    );

    return new PluralTranslatableMarkup($this->numTried,
      '@success, @failure, @skipped out of 1 total',
      '@success, @failure, @skipped out of @count total',
      [
        '@success' => $successMessage,
        '@failure' => $failureMessage,
        '@skipped' => $skippedMessage,
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function logFailure(): void {
    $this->numTried++;
    $this->numFailure++;
  }

  /**
   * {@inheritdoc}
   */
  public function logSkipped(): void {
    $this->numTried++;
    $this->numSkipped++;
  }

  /**
   * {@inheritdoc}
   */
  public function logSuccess(): void {
    $this->numTried++;
    $this->numSuccess++;
  }

}
