<?php

declare(strict_types=1);

namespace Drupal\Tests\scanner_fixer_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\scanner_fixer_api\Solution\SolutionInterface;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\CapableFailureFixerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\CapableSuccessfulFixerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\ExceptionFixerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\FiveItemsScannerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\IncapableFixerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\OneItemsScannerSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\TwoCapableSuccessfulFixersSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\TwoScannersSolution;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Solution\ZeroItemsScannerSolution;

/**
 * Tests that we can run solutions.
 *
 * @group scanner_fixer_api
 */
class SolutionTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['scanner_fixer_api_test', 'scanner_fixer_api'];

  /**
   * Test the solution that finds 1 item to fix, but fails to follow through.
   */
  public function testCapableFailureFixerSolution(): void {
    // Setup: Load an instance of the capable_failure_fixer plugin.
    $solution = $this->getSolutionFromPluginManager('capable_failure_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(CapableFailureFixerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 item to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 1 failures, 0 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(1, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution that finds 1 item and successfully fixes 1 item.
   */
  public function testCapableSuccessfulFixerSolution(): void {
    // Setup: Load an instance of the capable_successful_fixer plugin.
    $solution = $this->getSolutionFromPluginManager('capable_successful_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(CapableSuccessfulFixerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 item to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 1 successes, 0 failures, 0 skipped.
    $this->assertEquals(1, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution that finds 1 item but throws exception when fixing it.
   */
  public function testExceptionFixerSolution(): void {
    // Setup: Load an instance of the exception_fixer plugin.
    $solution = $this->getSolutionFromPluginManager('exception_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(ExceptionFixerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 items to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 1 failures, 0 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(1, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution whose scanner finds five items to fix.
   */
  public function testFiveItemsScannerSolution(): void {
    // Setup: Load an instance of the five_items_scanner plugin.
    $solution = $this->getSolutionFromPluginManager('five_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(FiveItemsScannerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 5 items to fix.
    $this->assertCount(5, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 0 failures, 0 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution that finds 1 item but cannot fix it.
   */
  public function testIncapableFixerSolution(): void {
    // Setup: Load an instance of the incapable_fixer plugin.
    $solution = $this->getSolutionFromPluginManager('incapable_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(IncapableFixerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 items to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 0 failures, 1 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(1, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution whose scanner finds one item to fix.
   */
  public function testOneItemsScannerSolution(): void {
    // Setup: Load an instance of the one_items_scanner plugin.
    $solution = $this->getSolutionFromPluginManager('one_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(OneItemsScannerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 items to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 0 failures, 0 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution that uses more than one fixer to get its jobs done.
   */
  public function testTwoCapableSuccessfulFixersSolution(): void {
    // Setup: Load an instance of the two_capable_successful_fixers plugin.
    $solution = $this->getSolutionFromPluginManager('two_capable_successful_fixers');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(TwoCapableSuccessfulFixersSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 1 items to fix.
    $this->assertCount(1, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 1 successes, 0 failures, 0 skipped.
    $this->assertEquals(1, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution that finds 1 item but throws exception when fixing it.
   */
  public function testTwoScannersSolution(): void {
    // Setup: Load an instance of the two_scanners_solution plugin.
    $solution = $this->getSolutionFromPluginManager('two_scanners_solution');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(TwoScannersSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 6 items to fix.
    $this->assertCount(6, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 6 successes, 0 failures, 0 skipped.
    $this->assertEquals(6, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Test the solution whose scanner finds nothing to fix.
   */
  public function testZeroItemsScannerSolution(): void {
    // Setup: Load an instance of the zero_items_scanner plugin.
    $solution = $this->getSolutionFromPluginManager('zero_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(ZeroItemsScannerSolution::class, $solution);

    // SUT: Call runScanners() on the plugin.
    $items = $solution->runScanners();

    // Assert: The solution finds 0 items to fix.
    $this->assertCount(0, $items);

    // SUT: Call runFixers() on the plugin with the items found earlier.
    $result = $solution->runFixers($items);

    // Assert: The solution reports 0 successes, 0 failures, 0 skipped.
    $this->assertEquals(0, $result->getNumberOfSuccesses());
    $this->assertEquals(0, $result->getNumberOfFailures());
    $this->assertEquals(0, $result->getNumberOfSkipped());
  }

  /**
   * Get a solution from the plugin manager.
   *
   * @param string $solutionId
   *   The ID of the solution to get.
   *
   * @return \Drupal\scanner_fixer_api\Solution\SolutionInterface
   *   An instance of that solution.
   */
  protected function getSolutionFromPluginManager(string $solutionId): SolutionInterface {
    return $this->container->get('plugin.manager.scanner_fixer_api.solution')
      ->createInstance($solutionId);
  }

}
