<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Solution;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\scanner_fixer_api\Fixer\FixerPluginManager;
use Drupal\scanner_fixer_api\Model\SolutionResultsInterface;
use Drupal\scanner_fixer_api\Model\SolutionStatCounter;
use Drupal\scanner_fixer_api\Scanner\ScannerPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A default, minimal implementation for a Scanner-Fixer Solution.
 */
abstract class SolutionBase implements SolutionInterface, ContainerFactoryPluginInterface {

  /**
   * A configuration array containing information about the plugin instance.
   *
   * @var array
   */
  protected array $configuration;

  /**
   * Plugin manager for fixers.
   *
   * @var \Drupal\scanner_fixer_api\Fixer\FixerPluginManager
   */
  protected FixerPluginManager $fixerPluginManager;

  /**
   * The plugin implementation definition.
   *
   * @var mixed
   */
  protected mixed $pluginDefinition;

  /**
   * The plugin ID for the plugin instance.
   *
   * @var string
   */
  protected string $pluginId;

  /**
   * Plugin manager for scanners.
   *
   * @var \Drupal\scanner_fixer_api\Scanner\ScannerPluginManager
   */
  protected ScannerPluginManager $scannerPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static();
    $instance->configuration = $configuration;
    $instance->pluginId = $plugin_id;
    $instance->pluginDefinition = $plugin_definition;
    $instance->fixerPluginManager = $container->get('plugin.manager.scanner_fixer_api.fixer');
    $instance->scannerPluginManager = $container->get('plugin.manager.scanner_fixer_api.scanner');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function runFixers(array $items = []): SolutionResultsInterface {
    $allResults = new SolutionStatCounter();

    $fixerPluginIds = $this->pluginDefinition['fixers'];
    $fixers = $this->fixerPluginManager->createInstancesFromClassNames($fixerPluginIds);
    foreach ($items as $item) {
      foreach ($fixers as $fixer) {
        try {
          if ($fixer->canFix($item)) {
            $result = $fixer->performFix($item);
            if ($result) {
              $allResults->logSuccess();
            }
            else {
              $allResults->logFailure();
            }
          }
          else {
            $allResults->logSkipped();
          }
        }
        catch (\Throwable) {
          $allResults->logFailure();
        }
      }
    }

    return $allResults;
  }

  /**
   * {@inheritdoc}
   */
  public function runScanners(): array {
    $items = [];

    $scannerPluginIds = $this->pluginDefinition['scanners'];
    $scanners = $this->scannerPluginManager->createInstancesFromClassNames($scannerPluginIds);
    foreach ($scanners as $scanner) {
      try {
        $items += $scanner->findProblemIds();
      }
      catch (\Throwable $t) {
        // No-op.
      }
    }

    return $items;
  }

}
