<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Drush\Commands;

use Consolidation\AnnotatedCommand\Attributes\Command;
use Consolidation\AnnotatedCommand\Attributes\Help;
use Consolidation\AnnotatedCommand\Attributes\Usage;
use Consolidation\OutputFormatters\StructuredData\ListDataInterface;
use Consolidation\OutputFormatters\StructuredData\UnstructuredListData;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\scanner_fixer_api\Solution\SolutionPluginManager;
use Drush\Attributes\Argument;
use Drush\Attributes\Bootstrap;
use Drush\Boot\DrupalBootLevels;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands related to Scanner-Fixer Solutions.
 */
final class SolutionCommands extends DrushCommands implements ContainerInjectionInterface {

  /**
   * Plugin manager for solutions.
   *
   * @var \Drupal\scanner_fixer_api\Solution\SolutionPluginManager
   */
  protected SolutionPluginManager $solutionPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->solutionPluginManager = $container->get('plugin.manager.scanner_fixer_api.solution');
    return $instance;
  }

  /**
   * List available solutions.
   *
   * @param array $options
   *   An associative array of command options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\ListDataInterface
   *   A list of available solutions.
   */
  #[Command(name: 'scanner-fixer:list-solutions')]
  #[Help(description: 'Lists available solutions.')]
  #[Usage(name: 'drush scanner-fixer:list-solutions', description: 'List available solutions.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function listSolutions(array $options = ['format' => 'json']): ListDataInterface {
    $scannerDefinitions = $this->solutionPluginManager->getDefinitions();
    return new UnstructuredListData($scannerDefinitions);
  }

  /**
   * Run an individual solution.
   *
   * @param string $solutionId
   *   The ID of the solution to run.
   * @param array $options
   *   An associative array of command options.
   */
  #[Command(name: 'scanner-fixer:run-solution')]
  #[Argument(name: 'solutionId', description: 'The ID of the solution to run')]
  #[Help(description: 'Runs a scanner-fixer solution.')]
  #[Usage(name: 'drush scanner-fixer:run-solution solutionId', description: 'Run the solution named solutionId.')]
  #[Bootstrap(level: DrupalBootLevels::FULL)]
  public function runSolution(string $solutionId, array $options = ['format' => 'json']): void {
    /** @var \Drupal\scanner_fixer_api\Solution\SolutionInterface $solution */
    $solution = $this->solutionPluginManager->createInstance($solutionId);
    $items = $solution->runScanners();
    $result = $solution->runFixers($items);
    $this->io()->note((string) $result->getSummaryString());
  }

}
