<?php

declare(strict_types=1);

namespace Drupal\scanner_fixer_api\Scanner;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\scanner_fixer_api\Annotation\Scanner;

/**
 * Plugin manager for scanners.
 */
class ScannerPluginManager extends DefaultPluginManager {

  /**
   * Constructs a ScannerPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   A cache backend for caching plugins of this type.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/ScannerFixer/Scanner',
      $namespaces,
      $module_handler,
      ScannerInterface::class,
      Scanner::class,
    );

    $this->alterInfo('scanner_fixer_scanner_info');
    $this->setCacheBackend($cache_backend, 'scanner_fixer_scanner_info_plugins');
  }

  /**
   * Instantiate a Scanner instance from a class name.
   *
   * @param string $className
   *   The class name of the Scanner object to instantiate.
   *
   * @return \Drupal\scanner_fixer_api\Scanner\ScannerInterface
   *   An instantiated Scanner plugin.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   Throws a plugin exception if a Scanner doesn't exist with the given class
   *   name.
   */
  public function createInstanceFromClassName(string $className): object {
    // Trim the leading slash.
    $className = ltrim($className, '\\');

    $matchingId = \array_reduce($this->getDefinitions(), static function ($carry, $definition) use ($className) {
      if ($definition['class'] === $className) {
        $carry = $definition['id'];
      }
      return $carry;
    }, '');

    return $this->createInstance($matchingId);
  }

  /**
   * Instantiate multiple Scanner instances from class names.
   *
   * @param string[] $classNames
   *   An array of Scanner plugin class names to instantiate.
   *
   * @return \Drupal\scanner_fixer_api\Scanner\ScannerInterface[]
   *   An array of instantiated Scanner plugins.
   */
  public function createInstancesFromClassNames(array $classNames): array {
    $instances = [];

    foreach ($classNames as $id) {
      try {
        $instances[$id] = $this->createInstanceFromClassName($id);
      }
      catch (PluginException) {
        // No-op.
      }
    }
    return $instances;
  }

}
