<?php

declare(strict_types=1);

namespace Drupal\Tests\scanner_fixer_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\scanner_fixer_api\Fixer\FixerInterface;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Fixer\CapableFailureFixer;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Fixer\CapableSuccessfulFixer;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Fixer\ExceptionFixer;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Fixer\IncapableFixer;

/**
 * Tests that we can run fixers.
 *
 * @group scanner_fixer_api
 */
class FixerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['scanner_fixer_api_test', 'scanner_fixer_api'];

  /**
   * A test item to try to fix.
   *
   * @var int
   */
  protected int $testItem;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Setup: Prepare a test item to try to fix.
    $this->testItem = \mt_rand();
  }

  /**
   * Test the fixer that says it can fix an item, but fails to fix it.
   */
  public function testCapableFailureFixer(): void {
    // Setup: Load an instance of the capable_failure_fixer plugin.
    $fixer = $this->getFixerFromPluginManager('capable_failure_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(CapableFailureFixer::class, $fixer);

    // SUT: Call canFix() on the test item.
    $canFix = $fixer->canFix($this->testItem);

    // Assert: The capable_failure_fixer should say it can perform the fix.
    $this->assertTrue($canFix);

    // SUT: Call performFix() on the item.
    $performFix = $fixer->performFix($this->testItem);

    // Assert: The capable_failure_fixer should say it didn't actually fix.
    $this->assertFalse($performFix);
  }

  /**
   * Test the fixer that says it can fix an item, then does so successfully.
   */
  public function testCapableSuccessfulFixer(): void {
    // Setup: Load an instance of the capable_successful_fixer plugin.
    $fixer = $this->getFixerFromPluginManager('capable_successful_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(CapableSuccessfulFixer::class, $fixer);

    // SUT: Call canFix() on the test item.
    $canFix = $fixer->canFix($this->testItem);

    // Assert: The capable_successful_fixer should say it can perform the fix.
    $this->assertTrue($canFix);

    // SUT: Call performFix() on the item.
    $performFix = $fixer->performFix($this->testItem);

    // Assert: The capable_successful_fixer should say it fixed the item.
    $this->assertTrue($performFix);
  }

  /**
   * Test the fixer that throws an exception when trying to fix the item.
   */
  public function testExceptionFixer(): void {
    // Setup: Load an instance of the exception_fixer plugin.
    $fixer = $this->getFixerFromPluginManager('exception_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(ExceptionFixer::class, $fixer);

    // SUT: Call canFix() on the test item.
    $canFix = $fixer->canFix($this->testItem);

    // Assert: The exception_fixer should say it can perform the fix.
    $this->assertTrue($canFix);

    // Assert: The exception_fixer should throw an exception on performFix().
    $this->expectException(\Exception::class);

    // SUT: Call performFix() on the item.
    $fixer->performFix($this->testItem);
  }

  /**
   * Test the fixer that says it cannot fix the item.
   */
  public function testIncapableFixer(): void {
    // Setup: Load an instance of the incapable_fixer plugin.
    $fixer = $this->getFixerFromPluginManager('incapable_fixer');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(IncapableFixer::class, $fixer);

    // SUT: Call canFix() on the test item.
    $canFix = $fixer->canFix($this->testItem);
    $this->assertFalse($canFix);

    // Assert: The incapable_fixer should throw an exception on performFix().
    $this->expectException(\LogicException::class);

    // SUT: Call performFix() on the item.
    $fixer->performFix($this->testItem);
  }

  /**
   * Get a fixer from the plugin manager.
   *
   * @param string $fixerId
   *   The ID of the fixer to get.
   *
   * @return \Drupal\scanner_fixer_api\Fixer\FixerInterface
   *   An instance of that fixer.
   */
  protected function getFixerFromPluginManager(string $fixerId): FixerInterface {
    return $this->container->get('plugin.manager.scanner_fixer_api.fixer')
      ->createInstance($fixerId);
  }

}
