<?php

declare(strict_types=1);

namespace Drupal\Tests\scanner_fixer_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\scanner_fixer_api\Scanner\ScannerInterface;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Scanner\FiveItemsScanner;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Scanner\OneItemsScanner;
use Drupal\scanner_fixer_api_test\Plugin\ScannerFixer\Scanner\ZeroItemsScanner;

/**
 * Tests that we can run scanners.
 *
 * @group scanner_fixer_api
 */
class ScannerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['scanner_fixer_api_test', 'scanner_fixer_api'];

  /**
   * Test the scanner that finds five items to fix.
   */
  public function testFiveItemsScanner(): void {
    // Setup: Load an instance of the five_items_scanner plugin.
    $scanner = $this->getScannerFromPluginManager('five_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(FiveItemsScanner::class, $scanner);

    // SUT: Call findProblemIds() on the plugin.
    $answer = $scanner->findProblemIds();

    // Assert: The five_items_scanner scanner should return 5 results.
    $this->assertCount(5, $answer);
  }

  /**
   * Test the scanner that finds one item to fix.
   */
  public function testOneItemsScanner(): void {
    // Setup: Load an instance of the one_items_scanner plugin.
    $scanner = $this->getScannerFromPluginManager('one_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(OneItemsScanner::class, $scanner);

    // SUT: Call findProblemIds() on the plugin.
    $answer = $scanner->findProblemIds();

    // Assert: The one_items_scanner scanner should return 1 results.
    $this->assertCount(1, $answer);
  }

  /**
   * Test the scanner that finds nothing to fix.
   */
  public function testZeroItemsScanner(): void {
    // Setup: Load an instance of the zero_items_scanner plugin.
    $scanner = $this->getScannerFromPluginManager('zero_items_scanner');

    // Assert: We got the plugin we expected.
    $this->assertInstanceOf(ZeroItemsScanner::class, $scanner);

    // SUT: Call findProblemIds() on the plugin.
    $answer = $scanner->findProblemIds();

    // Assert: The zero_items_scanner scanner should return 0 results.
    $this->assertCount(0, $answer);
  }

  /**
   * Get a scanner from the plugin manager.
   *
   * @param string $scannerId
   *   The ID of the scanner to get.
   *
   * @return \Drupal\scanner_fixer_api\Scanner\ScannerInterface
   *   An instance of that scanner.
   */
  protected function getScannerFromPluginManager(string $scannerId): ScannerInterface {
    return $this->container->get('plugin.manager.scanner_fixer_api.scanner')
      ->createInstance($scannerId);
  }

}
