<?php

namespace Drupal\Tests\scheduled_publish\Functional;

use Drupal\Component\Datetime\DateTimePlus;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\BrowserTestBase;

/**
 * Test Scheduled publish listing page.
 *
 * @group scheduled_publish
 */
class ScheduledPublishListingTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'field_ui',
    'scheduled_publish',
    'workflows',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $adminUser = $this->drupalCreateUser(['view any unpublished content']);
    $adminUser->addRole($this->createAdminRole('admin', 'admin'));
    $adminUser->save();
    $this->drupalLogin($adminUser);
  }

  /**
   * Test edit link on the scheduling listing page.
   *
   * @dataProvider cardinalityProvider
   */
  public function testEditLinkDependsOnCardinality($cardinality) {
    $this->prepareWorkflowConfiguration('article', $cardinality);

    $date = DateTimePlus::createFromTimestamp(strtotime('+1 hour'));

    $this->drupalGet('node/add/article');

    // Add scheduling for node.
    $this->submitForm([
      'field_publish[wrapper][moderation_state]' => 'published',
      'field_publish[wrapper][value][date]' => $date->format('Y-m-d'),
      'field_publish[wrapper][value][time]' => $date->format('H:i:s'),
    ], 'Schedule state change');

    // Save node with scheduling.
    $this->submitForm([
      'title[0][value]' => 'Test Scheduled Article',
    ], 'Save');

    $path = $this->getSession()->getCurrentUrl();
    preg_match('/node\/(\d+)/', $path, $matches);
    $nid = $matches[1];

    $this->drupalGet('admin/content/scheduled-publish');

    // Ensure that view with listing exist.
    $viewContainer = $this->getSession()->getPage()->find('css', '.views-element-container');
    $this->assertNotNull($viewContainer, 'Scheduled publish view container not found.');

    // Check that node exists in the listing.
    $this->assertStringContainsString(
      'Test Scheduled Article',
      $viewContainer->getText(),
      'The scheduled article title was not found in the listing.'
    );

    // Link to edit should exist.
    $link = $this->getSession()->getPage()->find(
      'xpath',
      sprintf('//a[contains(@href, "admin/content/scheduled-publish/edit/%d/")]', $nid),
    );
    $this->assertNotNull($link, 'Link not found.');
    $link->click();

    // Edit page open successfully.
    $this->assertSession()->pageTextContains('Edit status update for the "Test Scheduled Article (' . $nid . ')" node');
  }

  /**
   * Data provider for testEditLinkDependsOnCardinality().
   *
   * @return array[]
   *   Test cases with different cardinality.
   */
  public static function cardinalityProvider(): array {
    return [
      'single value' => [1],
      'unlimited' => [-1],
      'multiple values' => [3],
    ];
  }

  /**
   * Prepares a content type with a scheduled field and configures workflow.
   */
  protected function prepareWorkflowConfiguration(string $node_bundle = 'article', int $cardinality = 1): void {
    $this->createContentType(['type' => $node_bundle, 'name' => 'Article']);

    // Create Description field.
    FieldStorageConfig::create([
      'field_name' => 'field_publish',
      'type' => 'scheduled_publish',
      'entity_type' => 'node',
      'cardinality' => $cardinality,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_publish',
      'entity_type' => 'node',
      'bundle' => $node_bundle,
      'label' => 'Publish',
    ])->save();
    // Assign widget settings for the 'default' form mode.
    $this->container->get('entity_display.repository')
      ->getFormDisplay('node', $node_bundle)
      ->setComponent('field_publish', ['type' => 'scheduled_publish'])
      ->save();

    $this->drupalGet('admin/config/workflow/workflows/add');
    $this->submitForm([
      'edit-id' => 'editorial',
      'edit-label' => 'Editorial',
      'edit-workflow-type' => 'content_moderation',
    ], 'Save');
    $this->drupalGet('admin/config/workflow/workflows/manage/editorial/type/node');
    $this->submitForm([
      'edit-bundles-article' => TRUE,
    ], 'Save');
    $this->drupalGet('admin/config/workflow/workflows/manage/editorial');
  }

}
