<?php

namespace Drupal\Tests\scheduler\Functional;

use Drupal\node\NodeInterface;

/**
 * Tests revision options when Scheduler publishes or unpublishes content.
 *
 * @group scheduler
 */
class SchedulerRevisioningTest extends SchedulerBrowserTestBase {

  /**
   * Simulates the scheduled (un)publication of a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to schedule.
   * @param string $action
   *   The action to perform: either 'publish' or 'unpublish'. Defaults to
   *   'publish'.
   *
   * @return \Drupal\node\NodeInterface
   *   The updated node, after scheduled (un)publication via a cron run.
   */
  protected function schedule(NodeInterface $node, $action = 'publish') {
    // Simulate scheduling by setting the (un)publication date in the past and
    // running cron.
    $node->{$action . '_on'} = strtotime('-5 hour', $this->requestTime);
    $node->save();
    scheduler_cron();
    $this->nodeStorage->resetCache([$node->id()]);
    return $this->nodeStorage->load($node->id());
  }

  /**
   * Check if the number of revisions for a node matches a given value.
   *
   * @param int $nid
   *   The node id of the node to check.
   * @param string $value
   *   The value with which the number of revisions will be compared.
   * @param string $message
   *   The message to display along with the assertion.
   */
  protected function assertRevisionCount($nid, $value, $message = '') {
    // Because we are not deleting any revisions we can take a short cut and use
    // getLatestRevisionId() which will effectively be the number of revisions.
    $count = $this->nodeStorage->getLatestRevisionId($nid);
    $this->assertEquals($value, (int) $count, $message);
  }

  /**
   * Check if the latest revision log message of a node matches a given string.
   *
   * @param int $nid
   *   The node id of the node to check.
   * @param string $value
   *   The value with which the log message will be compared.
   * @param string $message
   *   The message to display along with the assertion.
   */
  protected function assertRevisionLogMessage($nid, $value, $message = '') {
    // Retrieve the latest revision log message for this node.
    $log_message = $this->database->select('node_revision', 'r')
      ->fields('r', ['revision_log'])
      ->condition('nid', $nid)
      ->orderBy('vid', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();
    $this->assertEquals($value, $log_message, $message);
  }

  /**
   * Tests the creation of new revisions on scheduling.
   */
  public function testRevisioning() {
    // Create a scheduled node that is not automatically revisioned.
    $created = strtotime('-2 day', $this->requestTime);
    $settings = [
      'type' => $this->type,
      'revision' => 0,
      'created' => $created,
    ];
    $node = $this->drupalCreateNode($settings);

    // Ensure nodes with past dates will be scheduled not published immediately.
    $this->nodetype->setThirdPartySetting('scheduler', 'publish_past_date', 'schedule')->save();

    // First test scheduled publication with revisioning disabled by default.
    $node = $this->schedule($node);
    $this->assertRevisionCount($node->id(), 1, 'No new revision is created by default when a node is published.');

    // Test scheduled unpublication.
    $node = $this->schedule($node, 'unpublish');
    $this->assertRevisionCount($node->id(), 1, 'No new revision is created by default when a node is unpublished.');

    // Enable revisioning.
    $this->nodetype->setThirdPartySetting('scheduler', 'publish_revision', TRUE)
      ->setThirdPartySetting('scheduler', 'unpublish_revision', TRUE)
      ->save();

    // Test scheduled publication with revisioning enabled.
    $node = $this->schedule($node);
    $this->assertRevisionCount($node->id(), 2, 'A new revision was created when revisioning is enabled.');
    $expected_message = sprintf('Published by Scheduler. The scheduled publishing date was %s.',
    $this->dateFormatter->format(strtotime('-5 hour', $this->requestTime), 'short'));
    $this->assertRevisionLogMessage($node->id(), $expected_message, 'The correct message was found in the node revision log after scheduled publishing.');

    // Test scheduled unpublication with revisioning enabled.
    $node = $this->schedule($node, 'unpublish');
    $this->assertRevisionCount($node->id(), 3, 'A new revision was created when a node was unpublished with revisioning enabled.');
    $expected_message = sprintf('Unpublished by Scheduler. The scheduled unpublishing date was %s.',
    $this->dateFormatter->format(strtotime('-5 hour', $this->requestTime), 'short'));
    $this->assertRevisionLogMessage($node->id(), $expected_message, 'The correct message was found in the node revision log after scheduled unpublishing.');
  }

  /**
   * Tests the 'touch' option to alter the node created date during publishing.
   */
  public function testAlterCreationDate() {
    // Ensure nodes with past dates will be scheduled not published immediately.
    $this->nodetype->setThirdPartySetting('scheduler', 'publish_past_date', 'schedule')->save();

    // Create a node with a 'created' date two days in the past.
    $created = strtotime('-2 day', $this->requestTime);
    $settings = [
      'type' => $this->type,
      'created' => $created,
      'status' => FALSE,
    ];
    $node = $this->drupalCreateNode($settings);
    // Show that the node is not published.
    $this->assertFalse($node->isPublished(), 'The node is not published.');

    // Schedule the node for publishing and run cron.
    $node = $this->schedule($node, 'publish');
    // Get the created date from the node and check that it has not changed.
    $created_after_cron = $node->created->value;
    $this->assertTrue($node->isPublished(), 'The node has been published.');
    $this->assertEquals($created, $created_after_cron, 'The node creation date is not changed by default.');

    // Set option to change the created date to match the publish_on date.
    $this->nodetype->setThirdPartySetting('scheduler', 'publish_touch', TRUE)->save();

    // Schedule the node again and run cron.
    $node = $this->schedule($node, 'publish');
    // Check that the created date has changed to match the publish_on date.
    $created_after_cron = $node->created->value;
    $this->assertEquals(strtotime('-5 hour', $this->requestTime), $created_after_cron, "With 'touch' option set, the node creation date is changed to match the publishing date.");

  }

}
