<?php

namespace Drupal\scheduler_field\Plugin\SchedulerField\Type;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\scheduler_field\SchedulerFieldTypePluginBase;

/**
 * Plugin implementation of the 'Publication' SchedulerFieldType.
 *
 * This plugin changes the publication status of entities by using the
 * EntityPublishedInterface. It works with any entity that implements
 * EntityPublishedInterface, regardless of the actual field name used
 * for publication (e.g., 'status', 'enabled', etc.).
 *
 * @SchedulerFieldType(
 *   id = "scheduler_field_type_publication",
 *   name = @Translation("Publication")
 * )
 */
class SchedulerFieldTypePublication extends SchedulerFieldTypePluginBase {

  /**
   * {@inheritDoc}
   */
  public function process(ContentEntityInterface $entity, FieldItemInterface $field_item) {
    $skip = FALSE;
    // If entity is unpublished, check if it has to be published thanks to date
    // values.
    if (!$entity->isPublished()) {
      $date_begin = $field_item->get('start_date')->getValue();
      $date_end = $field_item->get('end_date')->getValue();
      $current_date = new DrupalDateTime('now', DateTimeItemInterface::STORAGE_TIMEZONE);
      if ($date_begin < $current_date && (NULL === $date_end || $date_end > $current_date)) {
        $entity->setPublished();
        $entity->save();
        $skip = TRUE;
      }
    }
    // If entity is published, check if it has to be unpublished thanks to date
    // values.
    if (FALSE === $skip && $entity->isPublished()) {
      $date_end = $field_item->get('end_date')->getValue();
      $current_date = new DrupalDateTime('now', DateTimeItemInterface::STORAGE_TIMEZONE);
      if ($date_end < $current_date) {
        $entity->setUnpublished();
        $entity->save();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function processSchedulerQuery(SelectInterface $query, EntityStorageInterface $entity_storage, FieldStorageConfigInterface $field_storage): void {
    $field_name = $field_storage->getName();
    $entity_type_id = $entity_storage->getEntityTypeId();
    $field_table = $this->getFieldTableName($entity_type_id, $field_name);
    $data_table = $entity_storage->getDataTable() ?? $entity_storage->getBaseTable();
    $date_format = ($field_storage->getSetting('datetime_type') === 'datetime') ? 'Y-m-d\TH:i:s' : 'Y-m-d';

    // Get the published key from the entity type definition.
    $entity_type = $entity_storage->getEntityType();
    $published_key = $entity_type->getKey('published') ?: 'status';

    $now = new DrupalDateTime('now', DateTimeItemInterface::STORAGE_TIMEZONE);
    // Prepare query to filter entities to publish.
    $query_to_publish = $query->andConditionGroup();
    $end_date_condition = $query->orConditionGroup();
    $query_to_publish->condition("$field_table.{$field_name}_value", $now->format($date_format), '<')
      ->condition("$data_table.$published_key", 0);
    $end_date_condition->condition("$field_table.{$field_name}_end_value", $now->format($date_format), '>')
      ->condition("$field_table.{$field_name}_end_value", NULL, 'IS NULL');
    $query_to_publish->condition($end_date_condition);

    // Prepare query fo filter entities to unpublish.
    $query_to_unpublish = $query->andConditionGroup();
    $query_to_unpublish->condition("$field_table.{$field_name}_end_value", $now->format($date_format), '<')
      ->condition("$field_table.{$field_name}_end_value", NULL, 'IS NOT NULL')
      ->condition("$data_table.$published_key", 1);

    // Finally add filters to query.
    $query_or = $query->orConditionGroup();
    $query_or->condition($query_to_publish)
      ->condition($query_to_unpublish);
    $query->condition($query_or);
  }

  /**
   * {@inheritDoc}
   */
  public static function isAvailableForEntity(ContentEntityInterface $entity): bool {
    return $entity instanceof EntityPublishedInterface;
  }

  /**
   * {@inheritDoc}
   */
  public static function isAvailableForEntityType(string $entityTypeId): bool {
    $entity_type_manager = \Drupal::service('entity_type.manager');
    $entity_type = $entity_type_manager->getDefinition($entityTypeId, FALSE);

    // Check if the entity type has a published key defined.
    if ($entity_type && $entity_type->hasKey('published')) {
      return TRUE;
    }

    // Fallback: check if 'status' field exists for backward compatibility.
    $entity_field_manager = \Drupal::service('entity_field.manager');
    $field_definitions = $entity_field_manager->getBaseFieldDefinitions($entityTypeId);
    return isset($field_definitions['status']);
  }

}
