<?php

declare(strict_types=1);

namespace Drupal\Tests\scheduler_field\Kernel;

use Drupal\datetime\Plugin\Field\FieldType\DateTimeItem;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\scheduler_field_item_published_notstatus\Entity\SchedulerFieldEntityTestPublishedEnabled;

/**
 * Test scheduler_field field type publication in case published is not status.
 *
 * @group scheduler_field
 */
class SchedulerFieldPublicationNotStatusTest extends EntityKernelTestBase {

  /**
   * A field storage to use in this test class.
   *
   * @var \Drupal\field\Entity\FieldStorageConfig
   */
  protected $fieldStorage;

  /**
   * The field used in this test class.
   *
   * @var \Drupal\field\Entity\FieldConfig
   */
  protected $field;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'scheduler_field',
    'scheduler_field_item_published_notstatus',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('scheduler_field_test_pub_enabled');

    // Add a datetime range field.
    $this->fieldStorage = FieldStorageConfig::create([
      'field_name' => $this->randomMachineName(),
      'entity_type' => 'scheduler_field_test_pub_enabled',
      'type' => 'scheduler_field',
      'settings' => [
        'datetime_type' => DateTimeItem::DATETIME_TYPE_DATE,
        'scheduler_type' => 'scheduler_field_type_disabled',
      ],
      'cardinality' => 1,
    ]);
    $this->fieldStorage->save();

    $this->field = FieldConfig::create([
      'field_storage' => $this->fieldStorage,
      'bundle' => 'scheduler_field_test_pub_enabled',
      'required' => FALSE,
    ]);
    $this->field->save();

  }

  /**
   * Tests scheduled publication.
   */
  public function testPublication(): void {
    $field_name = $this->fieldStorage->getName();
    // Create an entity.
    $entity = SchedulerFieldEntityTestPublishedEnabled::create([
      'name' => $this->randomString(),
      $field_name => [
        'scheduler_type' => 'scheduler_field_type_publication',
        'value' => date('Y-m-d', strtotime('-2 day')),
        'end_value' => date('Y-m-d', strtotime('+1 day')),
      ],
    ]);
    $entity->setUnpublished();
    $entity->save();

    $this->reloadEntity($entity);
    $entity = SchedulerFieldEntityTestPublishedEnabled::load($entity->id());
    $this->assertNotTrue($entity->isPublished());

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    // First cron call put entity in queue.
    \Drupal::service('scheduler_field.cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(1, $queue->numberOfItems());

    // Second cron call unstack queue and process items.
    \Drupal::service('cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    $this->reloadEntity($entity);
    $entity = SchedulerFieldEntityTestPublishedEnabled::load($entity->id());
    $this->assertTrue($entity->isPublished());
  }

  /**
   * Tests scheduled unpublish.
   */
  public function testUnpublish(): void {
    $field_name = $this->fieldStorage->getName();
    // Create an entity.
    $entity = SchedulerFieldEntityTestPublishedEnabled::create([
      'name' => $this->randomString(),
      $field_name => [
        'scheduler_type' => 'scheduler_field_type_publication',
        'value' => date('Y-m-d', strtotime('-10 day')),
        'end_value' => date('Y-m-d', strtotime('-2 day')),
      ],
    ]);
    $entity->setPublished();
    $entity->save();

    $this->reloadEntity($entity);
    $entity = SchedulerFieldEntityTestPublishedEnabled::load($entity->id());
    $this->assertTrue($entity->isPublished());

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    // First cron call put entity in queue.
    \Drupal::service('scheduler_field.cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(1, $queue->numberOfItems());

    // Second cron call unstack queue and process items.
    \Drupal::service('cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    $this->reloadEntity($entity);
    $entity = SchedulerFieldEntityTestPublishedEnabled::load($entity->id());
    $this->assertNotTrue($entity->isPublished());
  }

}
