<?php

declare(strict_types=1);

namespace Drupal\Tests\scheduler_field\Kernel;

use Drupal\datetime\Plugin\Field\FieldType\DateTimeItem;
use Drupal\entity_test\Entity\EntityTestRevPub;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Test scheduler_field field type via API.
 *
 * @group scheduler_field
 */
class SchedulerFieldPublicationTest extends EntityKernelTestBase {

  /**
   * A field storage to use in this test class.
   *
   * @var \Drupal\field\Entity\FieldStorageConfig
   */
  protected $fieldStorage;

  /**
   * The field used in this test class.
   *
   * @var \Drupal\field\Entity\FieldConfig
   */
  protected $field;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'scheduler_field',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test_revpub');

    // Add a datetime range field.
    $this->fieldStorage = FieldStorageConfig::create([
      'field_name' => strtolower($this->randomMachineName()),
      'entity_type' => 'entity_test_revpub',
      'type' => 'scheduler_field',
      'settings' => [
        'datetime_type' => DateTimeItem::DATETIME_TYPE_DATE,
        'scheduler_type' => 'scheduler_field_type_disabled',
      ],
      'cardinality' => 1,
    ]);
    $this->fieldStorage->save();

    $this->field = FieldConfig::create([
      'field_storage' => $this->fieldStorage,
      'bundle' => 'entity_test_revpub',
      'required' => FALSE,
    ]);
    $this->field->save();

  }

  /**
   * Tests scheduled publication.
   */
  public function testPublication(): void {
    $field_name = $this->fieldStorage->getName();
    // Create an entity.
    $entity = EntityTestRevPub::create([
      'name' => $this->randomString(),
      $field_name => [
        'scheduler_type' => 'scheduler_field_type_publication',
        'value' => date('Y-m-d', strtotime('-2 day')),
        'end_value' => date('Y-m-d', strtotime('+1 day')),
      ],
    ]);
    $entity->setUnpublished();
    $entity->save();

    $this->reloadEntity($entity);
    $entity = EntityTestRevPub::load($entity->id());
    $this->assertNotTrue($entity->isPublished());

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    // First cron call put entity in queue.
    \Drupal::service('scheduler_field.cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(1, $queue->numberOfItems());

    // Second cron call unstack queue and process items.
    \Drupal::service('cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    $this->reloadEntity($entity);
    $entity = EntityTestRevPub::load($entity->id());
    $this->assertTrue($entity->isPublished());
  }

  /**
   * Tests scheduled unpublish.
   */
  public function testUnpublish(): void {
    $field_name = $this->fieldStorage->getName();
    // Create an entity.
    $entity = EntityTestRevPub::create([
      'name' => $this->randomString(),
      $field_name => [
        'scheduler_type' => 'scheduler_field_type_publication',
        'value' => date('Y-m-d', strtotime('-10 day')),
        'end_value' => date('Y-m-d', strtotime('-2 day')),
      ],
    ]);
    $entity->setPublished();
    $entity->save();

    $this->reloadEntity($entity);
    $entity = EntityTestRevPub::load($entity->id());
    $this->assertTrue($entity->isPublished());

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    // First cron call put entity in queue.
    \Drupal::service('scheduler_field.cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(1, $queue->numberOfItems());

    // Second cron call unstack queue and process items.
    \Drupal::service('cron')->run();

    $queue = \Drupal::service('queue')->get('scheduler_field_process');
    $this->assertEquals(0, $queue->numberOfItems());

    $this->reloadEntity($entity);
    $entity = EntityTestRevPub::load($entity->id());
    $this->assertNotTrue($entity->isPublished());
  }

  /**
   * Tests scheduled publication with datetime field and timezone handling.
   *
   * This test verifies that scheduling works correctly regardless of the
   * server's or user's timezone, by using dates stored in UTC.
   */
  public function testPublicationWithDatetimeAndTimezone(): void {
    // Create a datetime field storage.
    $datetime_field_storage = FieldStorageConfig::create([
      'field_name' => 'field_datetime_schedule',
      'entity_type' => 'entity_test_revpub',
      'type' => 'scheduler_field',
      'settings' => [
        'datetime_type' => DateTimeItem::DATETIME_TYPE_DATETIME,
        'scheduler_type' => 'scheduler_field_type_publication',
      ],
      'cardinality' => 1,
    ]);
    $datetime_field_storage->save();

    $datetime_field = FieldConfig::create([
      'field_storage' => $datetime_field_storage,
      'bundle' => 'entity_test_revpub',
      'required' => FALSE,
    ]);
    $datetime_field->save();

    // Save the original timezone to restore it later.
    $original_timezone = date_default_timezone_get();

    // Test with different timezones to ensure UTC handling is correct.
    $timezones = ['America/New_York', 'Europe/Paris', 'Asia/Tokyo'];

    foreach ($timezones as $test_timezone) {
      // Change the default timezone to simulate a user in a different timezone.
      date_default_timezone_set($test_timezone);

      // Set a start date 2 hours ago and end date in 1 day (in UTC).
      // Using gmdate ensures dates are in UTC regardless of current timezone.
      $start_date = gmdate('Y-m-d\TH:i:s', strtotime('-2 hours'));
      $end_date = gmdate('Y-m-d\TH:i:s', strtotime('+1 day'));

      $entity = EntityTestRevPub::create([
        'name' => $this->randomString() . ' - ' . $test_timezone,
        'field_datetime_schedule' => [
          'scheduler_type' => 'scheduler_field_type_publication',
          'value' => $start_date,
          'end_value' => $end_date,
        ],
      ]);
      $entity->setUnpublished();
      $entity->save();

      $this->reloadEntity($entity);
      $entity = EntityTestRevPub::load($entity->id());
      $this->assertFalse($entity->isPublished(), "Entity should be unpublished before cron in timezone: $test_timezone");

      // Run scheduler cron.
      \Drupal::service('scheduler_field.cron')->run();
      \Drupal::service('cron')->run();

      // Entity should now be published regardless of the timezone.
      $this->reloadEntity($entity);
      $entity = EntityTestRevPub::load($entity->id());
      $this->assertTrue($entity->isPublished(), "Entity should be published after cron in timezone: $test_timezone");

      // Clean up for next iteration.
      $entity->delete();
    }

    // Restore the original timezone.
    date_default_timezone_set($original_timezone);
  }

  /**
   * Tests scheduled unpublish with datetime field and timezone handling.
   *
   * This test verifies that unpublishing works correctly regardless of the
   * server's or user's timezone, by using dates stored in UTC.
   */
  public function testUnpublishWithDatetimeAndTimezone(): void {
    // Create a datetime field storage.
    $datetime_field_storage = FieldStorageConfig::create([
      'field_name' => 'field_datetime_unpublish',
      'entity_type' => 'entity_test_revpub',
      'type' => 'scheduler_field',
      'settings' => [
        'datetime_type' => DateTimeItem::DATETIME_TYPE_DATETIME,
        'scheduler_type' => 'scheduler_field_type_publication',
      ],
      'cardinality' => 1,
    ]);
    $datetime_field_storage->save();

    $datetime_field = FieldConfig::create([
      'field_storage' => $datetime_field_storage,
      'bundle' => 'entity_test_revpub',
      'required' => FALSE,
    ]);
    $datetime_field->save();

    // Save the original timezone to restore it later.
    $original_timezone = date_default_timezone_get();

    // Test with different timezones to ensure UTC handling is correct.
    $timezones = ['America/Los_Angeles', 'Europe/London', 'Australia/Sydney'];

    foreach ($timezones as $test_timezone) {
      // Change the default timezone to simulate a user in a different timezone.
      date_default_timezone_set($test_timezone);

      // Set dates in the past (in UTC).
      // Using gmdate ensures dates are in UTC regardless of current timezone.
      $start_date = gmdate('Y-m-d\TH:i:s', strtotime('-2 days'));
      $end_date = gmdate('Y-m-d\TH:i:s', strtotime('-1 hour'));

      $entity = EntityTestRevPub::create([
        'name' => $this->randomString() . ' - ' . $test_timezone,
        'field_datetime_unpublish' => [
          'scheduler_type' => 'scheduler_field_type_publication',
          'value' => $start_date,
          'end_value' => $end_date,
        ],
      ]);
      $entity->setPublished();
      $entity->save();

      $this->reloadEntity($entity);
      $entity = EntityTestRevPub::load($entity->id());
      $this->assertTrue($entity->isPublished(), "Entity should be published before cron in timezone: $test_timezone");

      // Run scheduler cron.
      \Drupal::service('scheduler_field.cron')->run();
      \Drupal::service('cron')->run();

      // Entity should now be unpublished regardless of the timezone.
      $this->reloadEntity($entity);
      $entity = EntityTestRevPub::load($entity->id());
      $this->assertFalse($entity->isPublished(), "Entity should be unpublished after cron in timezone: $test_timezone");

      // Clean up for next iteration.
      $entity->delete();
    }

    // Restore the original timezone.
    date_default_timezone_set($original_timezone);
  }

}
