<?php

declare(strict_types=1);

namespace Drupal\Tests\scheduler_field\Kernel;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormState;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItem;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\field\Kernel\FieldKernelTestBase;

/**
 * Test scheduler_field widget functionality.
 *
 * @group scheduler_field
 */
class SchedulerFieldWidgetTest extends FieldKernelTestBase {

  /**
   * A field storage to use in this test class.
   *
   * @var \Drupal\field\Entity\FieldStorageConfig
   */
  protected $fieldStorage;

  /**
   * The field used in this test class.
   *
   * @var \Drupal\field\Entity\FieldConfig
   */
  protected $field;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'system',
    'field',
    'text',
    'entity_test',
    'field_test',
    'datetime',
    'datetime_range',
    'scheduler_field',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Add a datetime range field.
    $this->fieldStorage = FieldStorageConfig::create([
      'field_name' => strtolower($this->randomMachineName()),
      'entity_type' => 'entity_test',
      'type' => 'scheduler_field',
      'settings' => [
        'datetime_type' => DateTimeItem::DATETIME_TYPE_DATETIME,
        'scheduler_type' => 'scheduler_field_type_publication',
      ],
      'cardinality' => 1,
    ]);
    $this->fieldStorage->save();

    $this->field = FieldConfig::create([
      'field_storage' => $this->fieldStorage,
      'bundle' => 'entity_test',
      'required' => FALSE,
    ]);
    $this->field->save();
  }

  /**
   * Tests automatic start date assignment when end date is set.
   */
  public function testSetStartDateFromEndDate(): void {
    $field_name = $this->fieldStorage->getName();

    // Create a widget instance.
    $widget = \Drupal::service('plugin.manager.field.widget')
      ->getInstance([
        'field_definition' => $this->field,
        'form_mode' => 'default',
        'prepare' => TRUE,
      ]);

    // Simulate setting only an end date (in the future) without start date.
    $future_date = new DrupalDateTime('+1 day');
    $element_values = [
      'value' => NULL,
      'end_value' => $future_date,
      'scheduler_type' => 'scheduler_field_type_publication',
    ];

    // Set up form state with the values.
    $form = [];
    $form_state = new FormState();
    $element = ['#parents' => [$field_name, 0]];
    $form_state->setValue([$field_name, 0], $element_values);

    // Call the validation method directly.
    $widget->setStartDateDependingOnEndDate($element, $form_state, $form);

    // Get the updated values.
    $updated_values = $form_state->getValue([$field_name, 0]);

    // Assert that start date was automatically set to now.
    $this->assertInstanceOf(DrupalDateTime::class, $updated_values['value']);
    $this->assertInstanceOf(DrupalDateTime::class, $updated_values['end_value']);

    // Verify that start date is before or equal to end date.
    $this->assertLessThanOrEqual($updated_values['end_value']->getTimestamp(), $updated_values['value']->getTimestamp());

    // Verify that start date is approximately now (within 5 seconds).
    $now = new DrupalDateTime();
    $this->assertLessThanOrEqual(5, abs($now->getTimestamp() - $updated_values['value']->getTimestamp()));
  }

  /**
   * Tests that start date is NOT set when end date is in the past.
   */
  public function testStartDateNotSetForPastEndDate(): void {
    $field_name = $this->fieldStorage->getName();

    // Create a widget instance.
    $widget = \Drupal::service('plugin.manager.field.widget')
      ->getInstance([
        'field_definition' => $this->field,
        'form_mode' => 'default',
        'prepare' => TRUE,
      ]);

    // Simulate setting only an end date in the past without start date.
    $past_date = new DrupalDateTime('-1 day');
    $element_values = [
      'value' => NULL,
      'end_value' => $past_date,
      'scheduler_type' => 'scheduler_field_type_publication',
    ];

    // Set up form state with the values.
    $form = [];
    $form_state = new FormState();
    $element = ['#parents' => [$field_name, 0]];
    $form_state->setValue([$field_name, 0], $element_values);

    // Call the validation method directly.
    $widget->setStartDateDependingOnEndDate($element, $form_state, $form);

    // Get the updated values.
    $updated_values = $form_state->getValue([$field_name, 0]);

    // Assert that start date was NOT automatically set (should still be NULL).
    $this->assertNull($updated_values['value']);
  }

  /**
   * Tests that existing start date is not overwritten.
   */
  public function testExistingStartDateNotOverwritten(): void {
    $field_name = $this->fieldStorage->getName();

    // Create a widget instance.
    $widget = \Drupal::service('plugin.manager.field.widget')
      ->getInstance([
        'field_definition' => $this->field,
        'form_mode' => 'default',
        'prepare' => TRUE,
      ]);

    // Simulate setting both start and end dates.
    $start_date = new DrupalDateTime('+1 hour');
    $end_date = new DrupalDateTime('+1 day');
    $element_values = [
      'value' => $start_date,
      'end_value' => $end_date,
      'scheduler_type' => 'scheduler_field_type_publication',
    ];

    // Set up form state with the values.
    $form = [];
    $form_state = new FormState();
    $element = ['#parents' => [$field_name, 0]];
    $form_state->setValue([$field_name, 0], $element_values);

    // Call the validation method directly.
    $widget->setStartDateDependingOnEndDate($element, $form_state, $form);

    // Get the updated values.
    $updated_values = $form_state->getValue([$field_name, 0]);

    // Assert that the original start date was not changed.
    $this->assertEquals($start_date->getTimestamp(), $updated_values['value']->getTimestamp());
  }

}
