<?php

namespace Drupal\schema_form\Attribute;

use Drupal\Component\Plugin\Attribute\Plugin;

/**
 * Defines the SchemaFormFieldType attribute for form field type plugins.
 *
 * The SchemaFormFieldType attribute is used to define plugins that convert
 * schema definitions into Drupal Form API elements. Each plugin handles
 * specific schema types and provides the logic for converting them into
 * appropriate form elements.
 *
 * Example usage:
 * @code
 * #[SchemaFormFieldType(
 *   id: "textfield",
 *   label: new TranslatableMarkup("Textfield"),
 *   description: new TranslatableMarkup("A text input field."),
 *   types: ["string", "email", "uri"],
 *   weight: 0
 * )]
 * class Textfield extends BaseSchemaFormFieldType {
 * }
 * @endcode
 */
#[\Attribute(\Attribute::TARGET_CLASS)]
class SchemaFormFieldType extends Plugin {

  /**
   * Constructs a SchemaFormFieldType attribute.
   *
   * @param string $id
   *   The plugin ID.
   * @param \Drupal\Core\Annotation\Translation $label
   *   The human-readable label of the field type.
   * @param \Drupal\Core\Annotation\Translation $description
   *   A description of what the field type does.
   * @param array $types
   *   An array of schema types this plugin can handle.
   * @param int $weight
   *   The weight determining the plugin's priority when multiple plugins
   *   can handle the same schema type.
   */
  public function __construct(
    public readonly string $id,
    public $label,
    public $description,
    public array $types = [],
    public int $weight = 0,
  ) {
  }

}
