<?php

namespace Drupal\schema_form\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\schema_form\SchemaForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration entities add and edit form based on the entity schema.
 *
 * Use this class as a form handler for the add and edit operations to get
 * the entity form automatically generated from the schema declaration.
 *
 * You can add customization to the generated form by creating a Schema Form
 * design for your custom config entity, using the `my_custom_entity_name.*`
 * schema name.
 */
class SchemaFormEntityForm extends EntityForm {

  /**
   * Constructs a SchemaFormEntityForm object.
   *
   * @param \Drupal\schema_form\SchemaForm $schemaForm
   *   The schema_form service.
   */
  public function __construct(
    protected SchemaForm $schemaForm,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('schema_form'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form += $this->schemaForm->getConfigEntityForm($this->entity);
    $form = parent::form($form, $form_state);
    return $form;
  }

  /**
   * Helper function to check whether the entity exists.
   */
  public function exist($id) {
    $entityType = $this->entity->getEntityTypeId();
    $entity = $this->entityTypeManager->getStorage($entityType)->load($id);
    return (bool) $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    // Redirect to the entity collection, if no custom destination is passed.
    $collectionLink = $this->entity->getEntityType()->getLinkTemplate('collection');
    $form_state->setRedirectUrl(Url::fromUri('internal:' . $collectionLink));
  }

}
