<?php

namespace Drupal\schema_form\Plugin\SchemaFormFieldType;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\schema_form\Plugin\SchemaFormFieldTypeInterface;
use Drupal\schema_form\SchemaForm;
use Drupal\schema_form\SchemaFormElementDto;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base class for Schema Form Field Type plugins.
 *
 * This abstract class implements common functionality for schema form field
 * type plugins, providing dependency injection and basic form element building
 * capabilities. Field type plugins should extend this class to inherit common
 * functionality while implementing their specific field type logic.
 */
abstract class BaseSchemaFormFieldType extends PluginBase implements SchemaFormFieldTypeInterface, ContainerFactoryPluginInterface {

  /**
   * The schema form service.
   *
   * @var \Drupal\schema_form\SchemaForm
   */
  protected SchemaForm $schemaForm;

  /**
   * Creates a new instance of the plugin.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container to pull out services used in the plugin.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   *
   * @return static
   *   Returns an instance of this plugin.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->schemaForm = $container->get('schema_form');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel() {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->pluginDefinition['description'];
  }

  /**
   * {@inheritdoc}
   */
  abstract public function buildTypedElement(SchemaFormElementDto $data): ?array;

  /**
   * {@inheritdoc}
   */
  public function buildElement(SchemaFormElementDto $data): ?array {
    $element = $this->buildTypedElement($data);
    if ($element !== NULL) {
      $element = $this->schemaForm->addBaseProperties($element, $data);
    }
    return $element;
  }

}
