<?php

namespace Drupal\schema_form\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\schema_form\SchemaFormElementDto;

/**
 * Interface for Schema Form Field Type plugins.
 *
 * Schema Form Field Type plugins are responsible for converting schema
 * definitions into Drupal Form API elements. Each plugin implementation handles
 * specific schema types (like string, email, boolean, etc.) and provides the
 * logic for converting them into appropriate form elements.
 *
 * @see \Drupal\schema_form\Plugin\SchemaFormFieldType\BaseSchemaFormFieldType
 * @see \Drupal\schema_form\Attribute\SchemaFormFieldType
 */
interface SchemaFormFieldTypeInterface extends PluginInspectionInterface {

  /**
   * Gets the human-readable label for this field type plugin.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The field type label.
   */
  public function getLabel();

  /**
   * Gets a description of what this field type plugin does.
   *
   * @return string|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The field type description.
   */
  public function getDescription();

  /**
   * Builds a typed form element from schema data.
   *
   * This method handles the specific implementation details for converting a
   * schema type into a form element. It is called by buildElement() after basic
   * setup.
   *
   * @param \Drupal\schema_form\SchemaFormElementDto $data
   *   The schema element DTO containing definition and metadata.
   *
   * @return array
   *   A Drupal Form API array for the specific element type.
   */
  public function buildTypedElement(SchemaFormElementDto $data): ?array;

  /**
   * Builds a form element from schema data.
   *
   * @param \Drupal\schema_form\SchemaFormElementDto $data
   *   The schema element DTO containing definition and metadata.
   *
   * @return array
   *   A Drupal Form API array representing the form element.
   */
  public function buildElement(SchemaFormElementDto $data): ?array;

}
