<?php

namespace Drupal\schema_form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Base class for configuration forms generated from schema definitions.
 *
 * This class extends Drupal's ConfigFormBase to provide automatic form
 * generation from configuration schemas. Forms extending this class will
 * automatically have their form elements generated based on the configuration
 * schema, while maintaining all the configuration form functionality like
 * saving and loading config values.
 *
 * Example usage:
 * @code
 * class MyConfigForm extends SchemaConfigFormBase {
 *   public function getEditableConfigNames() {
 *     return ['my_module.settings'];
 *   }
 * }
 * @endcode
 */
abstract class SchemaConfigFormBase extends ConfigFormBase {

  use SchemaFormBaseTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return $this->generateFormId();
  }

  /**
   * Generates a unique form ID based on the editable config names.
   *
   * @return string
   *   A unique form ID string.
   */
  protected function generateFormId() {
    return $this->toMachineName(implode('__', [
      'schema_config_form',
      ...$this->getEditableConfigNames(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Auto generate the form, if children do not declare any form elements.
    if ($this->schemaFormIsFormEmpty($form)) {
      $form = $this->getSchemaFormService()->getConfigsSchemaForm(
        $this->getEditableConfigNames(),
        $this->getSchemaFormDesignId()
      );
    }
    return parent::buildForm($form, $form_state);
  }

}
