<?php

namespace Drupal\schema_form;

use Drupal\schema_form\Exception\SchemaFormDesignInvalidException;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Serialization\Yaml;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\Core\Validation\BasicRecursiveValidatorFactory;
use Drupal\schema_form\Plugin\SchemaFormFieldTypeInterface;

/**
 * Service for processing and building schema-based forms.
 *
 * This service handles the conversion of configuration schema definitions into
 * Drupal Form API elements. It processes schema items and their metadata to
 * automatically generate form elements with appropriate properties and
 * validation.
 */
class SchemaForm {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Constant for marking schemas as configuration schemas.
   */
  public const SCHEMA_IS_CONFIG_SCHEMA = '_schema_form_is_config_schema';

  /**
   * Key used to store children elements in the form structure.
   *
   * @var string
   */
  const ELEMENT_CHILDREN_KEY = '#_schema_form_children';

  const DESIGN_ENTITY_PREFIX = 'schema_form.design.';

  /**
   * Constructs a SchemaForm object.
   *
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed configuration manager service.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typedDataManager
   *   The typed data manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Config\ConfigManagerInterface $configManager
   *   The configuration manager service.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $schemaFormFieldTypePluginManager
   *   The schema form field type plugin manager.
   * @param \Drupal\Core\Validation\BasicRecursiveValidatorFactory $validatorFactory
   *   The validator factory service.
   */
  public function __construct(
    protected TypedConfigManagerInterface $typedConfigManager,
    protected TypedDataManagerInterface $typedDataManager,
    protected ConfigFactoryInterface $configFactory,
    protected ConfigManagerInterface $configManager,
    protected PluginManagerInterface $schemaFormFieldTypePluginManager,
    protected BasicRecursiveValidatorFactory $validatorFactory,
  ) {
  }

  /**
   * Generates a form array from a schema definition.
   *
   * @param string $schemaName
   *   The name of the schema to build the form from.
   * @param array|null $defaultValues
   *   Default values for the form elements.
   * @param array|null $design
   *   Form design array with additional data to render schema elements.
   *
   * @return array
   *   A Form API array representing the schema.
   *
   * @throws \InvalidArgumentException
   *   When the specified schema is not found.
   */
  public function getSchemaForm(
    string $schemaName,
    ?array $defaultValues = NULL,
    ?array $design = NULL,
  ): array {
    $form = [];
    $schema = $this->getSchemaDefinition($schemaName);
    if ($schema['type'] === 'undefined') {
      throw new \InvalidArgumentException('Schema not found: ' . $schemaName);
    }

    $data = new SchemaFormElementDto(
      isFormRoot: TRUE,
      key: NULL,
      definition: $schema,
      design: $design,
      defaultValue: $defaultValues,
    );

    if ($schema[self::SCHEMA_IS_CONFIG_SCHEMA]) {
      // Attach the configuration object if loaded.
      $data->config = $this->configFactory->get($schemaName);

      // Check if the schema is a config entity schema and add the flag.
      if ($this->configManager->getEntityTypeIdByName($schemaName)) {
        $data->isConfigEntity = TRUE;
      }
    }

    $form = $this->createFormFromDefinition($data);

    if (!$data->config) {
      // Attach custom validation if the form is not a config form.
      $form['#validate'][] = [$this, 'validateForm'];
    }

    $form['#schema_form']['schema'] = $schemaName;

    return $form;
  }

  /**
   * Generates a form array for multiple schemas.
   *
   * @param array $configNames
   *   An array of configuration schema names.
   * @param string|null $design
   *   The design ID to use for building the form.
   *
   * @return array
   *   A Form API array representing the schemas.
   */
  public function getConfigsSchemaForm(array $configNames, ?string $design = NULL): array {
    foreach ($configNames as $configName) {
      if ($design) {
        $schemaForm = $this->getSchemaFormFromDesign($design, NULL, $configName);
      }
      else {
        $schemaForm = $this->getSchemaForm($configName);
      }
      // Return the direct form if only one schema is provided.
      if (count($configNames) === 1) {
        $form = $schemaForm;
      }
      else {
        $schema = $this->getSchemaDefinition($configName);
        $form[$this->prepareFormKey($configName)] = [
          '#type' => 'fieldset',
          '#tree' => TRUE,
          '#title' => $schema['label'],
          ...$schemaForm,
        ];
      }
    }
    return $form;
  }

  /**
   * Builds a form from the Schema Form customizations entity.
   *
   * @param string $id
   *   The id of the form design to use for building the form.
   * @param array|null $defaultValues
   *   Default values for the form elements.
   * @param string|null $schemaName
   *   The id of the Schema Form entity to use for building the form.
   *   If not provided, the default schema name from the design entity will be
   *   used.
   *
   * @return array
   *   The form definition as an array.
   *
   * @throws \Drupal\schema_form\Exception\SchemaFormDesignInvalidException
   *   When the form name is not valid or the schema is not found.
   */
  public function getSchemaFormFromDesign(
    string $id,
    ?array $defaultValues = NULL,
    ?string $schemaName = NULL,
  ): array {
    // @todo Maybe better rework to entity storage.
    $designEntity = $this->configFactory->get(self::DESIGN_ENTITY_PREFIX . $id);
    if (empty($designEntity->getRawData())) {
      throw new SchemaFormDesignInvalidException('The design entity with id "' . $id . '" is not found.');
    }
    $design = Yaml::decode($designEntity->get('design'));
    $schemaName ??= $designEntity->get('schema');
    if (!isset($schemaName)) {
      throw new SchemaFormDesignInvalidException('The schema name is not provided. Add the schema name to the design entity, or pass as an argument.');
    }
    return $this->getSchemaForm($schemaName, $defaultValues, $design);
  }

  /**
   * Builds a form from a configuration entity.
   *
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The configuration entity to build the form for.
   * @param string|null $designId
   *   (optional) The id of the Schema Form Design to apply.
   *   If not provided, the form will be built without any custom design.
   *
   * @return array
   *   The form definition as an array.
   */
  public function getConfigEntityForm(ConfigEntityInterface $entity, ?string $designId = NULL): array {
    /** @var \Drupal\Core\Config\Entity\ConfigEntityType $entityType    */
    $entityType = $entity->getEntityType();
    $configPrefix = $entityType->getConfigPrefix();
    $entitySchemaName = implode('.', [
      $configPrefix,
      $entity->id() ?? '*',
    ]);
    // The third_party_settings value is missing from the config entity
    // for some reason, so we need to fill it as the empty value manually.
    // Also fill the dependencies property, if empty.
    // @todo Make a better fix.
    $defaultValues = [
      'dependencies' => [],
      'third_party_settings' => [],
    ];
    $values = $entity->toArray() + $defaultValues;
    if (empty($designId)) {
      try {
        // Try to load the design by the config prefix.
        $designId = $configPrefix . '.edit_form';
        return $this->getSchemaFormFromDesign($designId, $values, $entitySchemaName);
      }
      catch (SchemaFormDesignInvalidException) {
        // If not found, render the form from schema.
        return $this->getSchemaForm($entitySchemaName, $values);
      }
    }
    else {
      return $this->getSchemaFormFromDesign($designId, $values, $entitySchemaName);
    }

  }

  /**
   * Prepares a form key by replacing invalid characters.
   *
   * @param string $name
   *   The original name.
   *
   * @return string
   *   The prepared form key.
   */
  protected function prepareFormKey($name) {
    return preg_replace('/[^a-zA-Z0-9_]/', '_', $name);
  }

  /**
   * Adds base properties to a form element.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\schema_form\SchemaFormElementDto $data
   *   The schema form element data transfer object.
   *
   * @return array
   *   The form element with base properties added.
   */
  public function addBaseProperties(array $element, SchemaFormElementDto $data): array {
    $definition = $data->definition;
    if ($value = $this->getMetadataFromDefinition(['label', 'title'], $definition)) {
      $element['#title'] = $value;
    }
    if ($value = $this->getMetadataFromDefinition('description', $definition)) {
      $element['#description'] = $value;
    }
    // @todo Do not add default value for fieldsets.
    $element['#default_value'] = $data->defaultValue;

    // Mark the form field as required if have specific constraints.
    if (!empty($definition['constraints'])) {
      if (
        array_intersect([
          'NotBlank',
          'NotNull',
        ], array_keys($definition['constraints']))) {
        $element['#required'] = TRUE;
      }
    }
    if ($elementOverrides = $this->getDefinitionThirdPartySetting($definition, 'form_element')) {
      $element = array_merge($element, $elementOverrides);
    }
    $this->mergeDefinitionHashElements($element, $definition);
    // Apply the form data the highest priority.
    if ($formElementData = $data->design['form_element'] ?? NULL) {
      $element = array_merge($element, $formElementData);
    }

    // Apply translation for common properties.
    foreach (['#title', '#description'] as $property) {
      if (
        isset($element[$property])
        && !empty($element[$property])
        && is_string($element[$property])
      ) {
        // The value is quite static, because imported from the schema.
        // Ignoring the phpcs rule for dynamic strings for this case.
        // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
        $element[$property] = $this->t($element[$property]);
      }
    }
    if (isset($element['#options'])) {
      foreach ($element['#options'] as $key => $value) {
        if (!empty($value) && is_string($value)) {
          // The value is quite static, because imported from the schema.
          // Ignoring the phpcs rule for dynamic strings for this case.
          // phpcs:ignore Drupal.Semantics.FunctionT.NotLiteralString
          $element['#options'][$key] = $this->t($value);
        }
      }
    }

    return $element;
  }

  /**
   * Retrieves a specific third-party setting from a schema definition.
   *
   * @param array $definition
   *   The schema definition array.
   * @param string $key
   *   The key of the third-party setting to retrieve.
   *
   * @return mixed
   *   The value of the third-party setting if it exists, or NULL otherwise.
   */
  public function getDefinitionThirdPartySetting(array $definition, string $key): mixed {
    return $definition['third_party_settings']['schema_form'][$key] ?? NULL;
  }

  /**
   * Gets a third party setting from a form element.
   *
   * @param array $element
   *   The form element.
   * @param string $key
   *   The key of the setting to retrieve.
   *
   * @return mixed
   *   The value of the setting if it exists, NULL otherwise.
   */
  public function getElementThirdPartySetting(array $element, string $key): mixed {
    return $element['#third_party_settings']['schema_form'][$key] ?? NULL;
  }

  /**
   * Sets a third party setting for a form element.
   *
   * @param array &$element
   *   The form element to modify.
   * @param string $key
   *   The key of the setting to set.
   * @param mixed $data
   *   The value to set.
   */
  public function setElementThirdPartySetting(array &$element, string $key, mixed $data): void {
    $element['#third_party_settings']['schema_form'][$key] = $data;
  }

  /**
   * Merges hash elements from definition into element array.
   *
   * @param array &$element
   *   The form element to modify.
   * @param array $definition
   *   The schema definition containing hash elements.
   */
  public function mergeDefinitionHashElements(array &$element, array $definition): void {
    foreach ($definition as $key => $value) {
      if (str_starts_with($key, '#')) {
        $element[$key] = $value;
      }
    }
  }

  /**
   * Retrieves metadata from a schema definition.
   *
   * @param string|array $names
   *   The metadata names to retrieve.
   * @param array $definition
   *   The schema definition.
   *
   * @return mixed
   *   The metadata value, or NULL if not found.
   */
  public function getMetadataFromDefinition(string|array $names, array $definition): mixed {
    if (is_string($names)) {
      $names = [$names];
    }
    foreach ($names as $name) {
      $value =
        $this->getDefinitionThirdPartySetting($definition, $name)
        ?? $definition[$name]
        ?? $definition['#' . $name]
        ?? NULL;
      if ($value) {
        return $value;
      }
    }
    return NULL;
  }

  /**
   * Adds children elements to a form element.
   *
   * @param array $element
   *   The form element.
   * @param array $children
   *   The children elements.
   *
   * @return array
   *   The form element with children added.
   */
  public function addChildrenElements(array $element, array $children): array {
    $element[self::ELEMENT_CHILDREN_KEY] = $children;
    return $element;
  }

  /**
   * Get a human-readable override message.
   *
   * @param mixed $value
   *   The overridden value.
   *
   * @return string
   *   A human-readable message describing how and why the config value is
   *   overridden.
   */
  public function getOverriddenMessage($value = NULL): string {
    return $value
      ? $this->t("This config value is overridden as <code>@value</code>", ['@value' => $value])
      : $this->t("This config value is overridden.");
  }

  /**
   * Loads a plugin for a form element based on the schema definition.
   *
   * @param \Drupal\schema_form\SchemaFormElementDto $data
   *   The schema form element data transfer object.
   *
   * @return \Drupal\schema_form\Plugin\SchemaFormFieldTypeInterface|null
   *   The loaded plugin, or NULL if no suitable plugin is found.
   */
  private function loadPluginForFormElement(SchemaFormElementDto $data): ?SchemaFormFieldTypeInterface {
    $pluginDefinitions = $this->schemaFormFieldTypePluginManager->getDefinitions();
    uasort($pluginDefinitions, fn ($a, $b) => ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0));

    foreach ($pluginDefinitions as $pluginId => $pluginDefinition) {
      if (
        isset($pluginDefinition['types'])
        && in_array($data->definition['type'], $pluginDefinition['types'], TRUE)
      ) {
        $plugin = $this->schemaFormFieldTypePluginManager->createInstance($pluginId, ['data' => $data]);
        break;
      }
    }
    return $plugin ?? NULL;
  }

  /**
   * Generates a form element from a schema definition.
   *
   * @param \Drupal\schema_form\SchemaFormElementDto $data
   *   The schema form element data transfer object.
   *
   * @return array|null
   *   The form element, or NULL if no suitable plugin is found.
   *
   * @throws \RuntimeException
   *   When no suitable plugin is found for the schema type.
   */
  private function createFormFromDefinition(SchemaFormElementDto $data) {
    // Skip special properties.
    if ($data->key && str_starts_with($data->key ?? '', '_')) {
      return NULL;
    }
    if ($data->isConfigEntity) {
      // Config entities have some special properties that should not be
      // exposed in the form.
      // @todo Try to find a better way to hide them.
      $hiddenEntityFormFields = [
        'uuid',
        'dependencies',
        'third_party_settings',
      ];
      if (in_array($data->key, $hiddenEntityFormFields)) {
        return NULL;
      }
    }
    if ($data->design['skip'] ?? NULL) {
      return NULL;
    }

    // Process the mapping type for non-known types.
    if (!$plugin = $this->loadPluginForFormElement($data)) {
      // If no suitable plugin was found, try to load the schema with the
      // name matching the type, and merge it into the current definition.
      try {
        $referencedSchema = $this->getSchemaDefinition($data->definition['type']);
      }
      catch (\RuntimeException) {
        $referencedSchema = NULL;
      }
      if ($referencedSchema) {
        $dataReferenced = clone $data;
        // Merging the current definition over the referenced one.
        $dataReferenced->definition = $referencedSchema;
        $dataReferenced->mergeDefinition($data->definition);
        $dataReferenced->design = $data->design ?? NULL;
        $data = $dataReferenced;
      }

      // Process the mapping type for non-known types, if the mapping exists.
      if (isset($data->definition['mapping'])) {
        $plugin = $this->schemaFormFieldTypePluginManager->createInstance('mapping', ['data' => $data]);
      }
      else {
        throw new \RuntimeException("No suitable plugin found for type: {$data->definition['type']}");
      }

    }

    if ($data->isFormRoot) {
      $element = [];
      $children = $data->definition['mapping'];
    }
    else {
      $element = $plugin->buildElement($data);
      if (
        $element !== NULL
        && isset($element[self::ELEMENT_CHILDREN_KEY])
        && $element['#type'] !== 'value'
      ) {
        $children = $element[self::ELEMENT_CHILDREN_KEY];
        unset($element[self::ELEMENT_CHILDREN_KEY]);
      }
      else {
        $children = [];
      }
    }

    // Support non-standard children elements.
    foreach ($children as $subKey => $subDefinition) {
      $itemData = clone $data;
      $itemData->key = $subKey;
      $itemData->definition = $subDefinition;
      $itemData->design = $data->design['mapping'][$subKey] ?? NULL;
      $itemData->parents = $data->key
        ? [...$data->parents ?? [], $data->key]
        : NULL;
      $itemData->defaultValue = $data->defaultValue[$subKey] ?? NULL;
      $itemData->isFormRoot = FALSE;
      $element[$subKey] = $this->createFormFromDefinition($itemData);
    }

    if (
      $element !== NULL
      && !$data->isFormRoot
      && $data->config
    ) {
      $element['#config_target'] = $data->config->getName() . ':' . $data->getKeyPath();
      // Add the default value and override properties only for valuable types.
      if ($data->definition['type'] != 'mapping') {
        $valueOriginal = $data->config->getOriginal($data->getKeyPath(), FALSE);
        $valueOverridden = $data->config->get($data->getKeyPath());
        $element['#default_value'] = $data->defaultValue ?? $valueOriginal;
        if ($valueOverridden != $valueOriginal) {
          // If the element already has the description, convert it to array
          // to add a new line as a suffix.
          if (isset($element['#description']) && !is_array($element['#description'])) {
            $element['#description'] = [$element['#description']];
          }
          if ($element['#type'] === 'password') {
            $valueOverridden = NULL;
          }
          $element['#description']['schema_form_note'] = $this->getOverriddenMessage($valueOverridden);
          $element['#description'] = implode(' ', array_filter($element['#description']));
        }
      }
    }

    return $element;
  }

  /**
   * Retrieves a schema definition by name.
   *
   * @param string $name
   *   The name of the schema.
   *
   * @return array
   *   The schema definition.
   *
   * @throws \RuntimeException
   *   When the schema definition is not found.
   */
  public function getSchemaDefinition(string $name) {
    $schema = $this->typedConfigManager->getDefinition($name);
    if ($schema['type'] === 'undefined') {
      throw new \RuntimeException("Failed to load the schema with name: $name");
    }

    // The original schema type got replaced in the getDefinition() method to
    // the schema name, so we have no direct ways to determine if the schema
    // is a config schema or not.
    // Checking by the config-specific mapping keys.
    // @see https://www.drupal.org/project/drupal/issues/3477363
    // @todo Invent a better way.
    if (($schema['mapping']['_core']['type'] ?? NULL) == '_core_config_info') {
      $schema[self::SCHEMA_IS_CONFIG_SCHEMA] = TRUE;
    }
    else {
      $schema[self::SCHEMA_IS_CONFIG_SCHEMA] = FALSE;
    }

    return $schema;
  }

  /**
   * Validates the form based on the schema definition.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $schemaName = $form['#schema_form']['schema'];

    // Clone the form state to avoid modifying the original by cleanValues().
    $formStateClone = clone $form_state;
    $values = $formStateClone->cleanValues()->getValues();
    if ($form['#parents']) {
      $values = NestedArray::getValue($values, $form['#parents']);
    }

    $typedData = $this->typedConfigManager->createFromNameAndData($schemaName, $values);
    $violations = $typedData->validate();

    foreach ($violations as $violation) {
      $formPropertyPath = $violation->getPropertyPath();
      $statePropertyPath = $form['#parents'] ? implode('.', $form['#parents']) . '.' . $formPropertyPath : $formPropertyPath;
      $statePropertyPathAsName = str_replace('.', '][', $statePropertyPath);
      $form_state->setErrorByName($statePropertyPathAsName, $violation->getMessage());
    }
  }

  /**
   * Creates a TypedData object from the provided values and schema name.
   *
   * @param array $values
   *   The values to set in the TypedData object.
   * @param string $schemaName
   *   The name of the schema to use for creating the TypedData object.
   *
   * @return \Drupal\Core\TypedData\TypedDataInterface
   *   The created TypedData object.
   */
  public function createTypedValues(array $values, string $schemaName): TypedDataInterface {
    $typedData = $this->typedConfigManager->createFromNameAndData($schemaName, $values);
    return $typedData;
  }

}
