<?php

namespace Drupal\schema_form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\TraversableTypedDataInterface;

/**
 * Provides a base form class for schema-based Drupal forms.
 *
 * This abstract class provides the foundation for creating forms that are
 * automatically generated from schema definitions. It extends Drupal's FormBase
 * and uses the SchemaFormBaseTrait to provide schema form functionality.
 *
 * Example usage:
 * @code
 * class MyForm extends SchemaFormBase {
 *   public function getSchemaFormSchema(): string {
 *     return 'my_module.my_form';
 *   }
 *
 *   public function processSubmittedValues(array $values) {
 *     // Handle form submitted values.
 *   }
 * }
 * @endcode
 */
abstract class SchemaFormBase extends FormBase {
  use SchemaFormBaseTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return $this->toMachineName(
      $this->getSchemaFormDesignId()
      ?? $this->getSchemaFormSchema()
      ?? 'schema_form_default_form'
    );
  }

  /**
   * Returns the schema name used to build this form.
   *
   * Override this method to specify which schema definition should be used to
   * generate the form structure. The schema name should correspond to a valid
   * schema definition in your module's schema files.
   *
   * @return string|null
   *   The schema name, or NULL. Use returning NULL to generate manually a
   *   customized form using getSchemaForm() and your custom processing.
   */
  protected function getSchemaFormSchema(): ?string {
    return NULL;
  }

  /**
   * Gets the default values for the schema form fields.
   *
   * Override this method to provide default values for form fields defined in
   * the schema. The returned array should have keys matching the schema field
   * names and corresponding default values.
   *
   * @return array
   *   An associative array of default values, where keys are field names and
   *   values are their default values.
   */
  protected function getSchemaFormDefaultValues(): array {
    return [];
  }

  /**
   * Gets the form structure from a schema definition.
   *
   * @param string $schemaName
   *   The name of the schema to build the form from.
   * @param array|null $defaultValues
   *   Optional array of default values for the form fields. If not provided,
   *   defaults to values from getSchemaFormDefaultValues().
   *
   * @return array
   *   A Form API array representing the schema-based form structure.
   */
  protected function getSchemaForm(string $schemaName, ?array $defaultValues = NULL): array {
    if (empty($defaultValues)) {
      $defaultValues = $this->getSchemaFormDefaultValues();
    }
    return $this->getSchemaFormService()->getSchemaForm($schemaName, $defaultValues);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if ($designId = $this->getSchemaFormDesignId()) {
      $form += $this->getSchemaFormService()->getSchemaFormFromDesign($designId);
    }
    elseif ($schema = $this->getSchemaFormSchema()) {
      $form += $this->getSchemaForm($schema);
    }

    // Add standard form actions if not already present.
    if (!isset($form['actions'])) {
      $form['actions'] = [
        '#type' => 'actions',
      ];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Submit'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $schemaFormService = $this->getSchemaFormService();
    // We should clone the form_state to avoid modifying the original one
    // when use cleanValues().
    $formStateCloned = clone $form_state;
    $values = $formStateCloned->cleanValues()->getValues();
    $schema = $this->getSchemaFormSchema();
    $valuesTypedData = $schemaFormService->createTypedValues($values, $schema);
    $this->processSubmittedValues($valuesTypedData, $form_state);
  }

  /**
   * Processes the submitted values after validation.
   *
   * This method can be overridden in subclasses to handle the submitted values
   * after they have been validated. It receives the validated values as a
   * TraversableTypedDataInterface object.
   *
   * @param \Drupal\Core\TypedData\TraversableTypedDataInterface $values
   *   The validated form submission values.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  protected function processSubmittedValues(TraversableTypedDataInterface $values, FormStateInterface $form_state): void {
    // Put here all the custom handling of the already validated form submission
    // values. If you process them in the submitForm() method directly, you can
    // keep this method empty.
  }

}
