<?php

namespace Drupal\schema_form;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides base schema form functionality and service management.
 *
 * This trait provides common functionality for schema form classes, including
 * service injection and form building utilities.
 */
trait SchemaFormBaseTrait {

  /**
   * The Schema Form service.
   *
   * @var \Drupal\schema_form\SchemaForm
   */
  protected SchemaForm $schemaForm;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    // Soft adding the container.
    if (
      property_exists($instance, 'container')
      && !isset($instance->container)
    ) {
      $instance->container = $container;
    }
    return $instance;
  }

  /**
   * Gets the Schema Form service, loading it from container if needed.
   *
   * @return \Drupal\schema_form\SchemaForm
   *   The Schema Form service.
   */
  protected function getSchemaFormService(): SchemaForm {
    if (!isset($this->schemaForm)) {
      $container = $this->container ?? \Drupal::getContainer();
      $this->schemaForm = $container->get('schema_form');
    }
    return $this->schemaForm;
  }

  /**
   * Determines if a form array is empty of real elements.
   *
   * A form is considered empty if it contains no elements other than properties
   * (keys starting with #).
   *
   * @param array $form
   *   The form array to check.
   *
   * @return bool
   *   TRUE if the form has no real elements, FALSE otherwise.
   */
  protected function schemaFormIsFormEmpty($form): bool {
    $formRealElements = array_filter(array_keys($form), function ($value) {
      return strpos($value, '#') !== 0;
    });
    return empty($formRealElements);
  }

  /**
   * Returns the id of the Schema Form Design entity to render form from.
   *
   * Override this method to specify which schema design should be used to
   * generate the form structure.
   *
   * @return string|null
   *   The schema_form_design entity id, or NULL. Use returning NULL to generate
   *   the default representation of the schema.
   */
  protected function getSchemaFormDesignId(): ?string {
    return NULL;
  }

  /**
   * Generates a machine name from a given string.
   *
   * This method converts the input string to lowercase and replaces any
   * non-alphanumeric characters with underscores.
   *
   * @param string|null $name
   *   The input string to convert. If NULL, an empty string is returned.
   *
   * @return string
   *   The generated machine name.
   */
  protected function toMachineName(?string $name = NULL): string {
    return strtolower(preg_replace('/[^a-z0-9_]+/', '_', $name));
  }

}
