<?php

namespace Drupal\schema_form;

use Drupal\Core\Config\ImmutableConfig;

/**
 * Data transfer object for schema form elements.
 *
 * This class encapsulates the data needed to build a form element from a schema
 * definition, including the element's definition, default values, path, and any
 * associated validation constraints.
 */
class SchemaFormElementDto {

  /**
   * The element's default value.
   *
   * @var mixed
   */
  public mixed $defaultValue = NULL;

  /**
   * An array of parent keys defining this element's location in the form.
   *
   * @var array|null
   */
  public ?array $parents = NULL;

  /**
   * Constructs a new SchemaFormElementDto object.
   *
   * @param array $definition
   *   The schema definition for this form element.
   * @param string $key
   *   The key identifying this form element.
   * @param mixed $defaultValue
   *   (optional) The default value for this form element.
   * @param array|null $design
   *   (optional) The form design data.
   * @param array|null $parents
   *   (optional) An array of parent keys defining this element's location in
   *   the form structure.
   * @param \Drupal\Core\Config\ImmutableConfig|null $config
   *   (optional) The configuration object associated with this form element.
   * @param bool|null $isConfigEntity
   *   (optional) Indicates if this element is a configuration entity.
   * @param bool|null $isFormRoot
   *   (optional) Indicates if this element is the root of a form.
   */
  public function __construct(
    public ?array $definition,
    public ?string $key = NULL,
    mixed $defaultValue = NULL,
    public ?array $design = NULL,
    ?array $parents = NULL,
    public ?ImmutableConfig $config = NULL,
    public ?bool $isConfigEntity = FALSE,
    public ?bool $isFormRoot = FALSE,
  ) {
    $this->defaultValue = $defaultValue;
    $this->parents = $parents;
  }

  /**
   * Recursively merges two arrays.
   *
   * @param array $array1
   *   The first array.
   * @param array $array2
   *   The second array.
   *
   * @return array
   *   The merged array.
   */
  private function recursiveArrayMerge(array $array1, array $array2): array {
    foreach ($array2 as $key => $value) {
      if (is_array($value) && isset($array1[$key]) && is_array($array1[$key])) {
        $array1[$key] = $this->recursiveArrayMerge($array1[$key], $value);
      }
      else {
        $array1[$key] = $value;
      }
    }
    return $array1;
  }

  /**
   * Merges additional schema definition properties into the current definition.
   *
   * This method allows for extending or overriding the current schema
   * definition with additional properties, which is useful when processing
   * referenced schemas or applying customizations.
   *
   * @param array $definition
   *   The additional schema definition properties to merge.
   */
  public function mergeDefinition(array $definition): self {
    $this->definition = $this->recursiveArrayMerge($this->definition, $definition);
    return $this;
  }

  /**
   * Gets the complete key path for this form element.
   *
   * Generates a dot-separated string representing the full path to this element
   * in the form structure, including all parent keys.
   *
   * @return string
   *   The complete configuration key path (e.g. 'parent.child.key').
   */
  public function getKeyPath(): string {
    return $this->parents
      ? implode('.', $this->parents) . '.' . $this->key
      : $this->key ?? '';
  }

}
