<?php

declare(strict_types=1);

namespace Drupal\schema_form_test\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Form\FormStateInterface;
use Drupal\schema_form\SchemaForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Schema Form add and edit forms.
 */
class BlockContentTypeForm extends EntityForm {

  /**
   * Constructs a SchemaFormEntityForm object.
   *
   * @param \Drupal\schema_form\SchemaForm $schemaForm
   *   The schema_form service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    protected SchemaForm $schemaForm,
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandler $moduleHandler,
  ) {
    $this->setEntityTypeManager($entityTypeManager);
    $this->setModuleHandler($moduleHandler);

    // @todo Use getFormArgument() or getFormObject() instead.
    $formArgument = $this->getRouteMatch()->getRouteObject()->getDefault('_entity_form');
    [$entityTypeId] = explode('.', $formArgument);
    $entity = $this->getEntityFromRouteMatch($this->routeMatch, $entityTypeId);
    $this->setEntity($entity);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('schema_form'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $schemaForm = $this->schemaForm->getConfigEntityForm($this->entity);
    $form = parent::form($schemaForm, $form_state);
    return $form;
  }

  /**
   * Helper function to check whether the entity exists.
   */
  public function exist($id) {
    $entity = $this->entityTypeManager->getStorage('schema_form')->getQuery()
      ->accessCheck(FALSE)
      ->condition('id', $id)
      ->execute();
    return (bool) $entity;
  }

}
