<?php

declare(strict_types=1);

namespace Drupal\Tests\schema_form\Kernel;

use Drupal\schema_form_test\Form\CoreSystemPerformanceForm;
use Drupal\schema_form_test\Form\MyApiConfigForm;

/**
 * Tests the schema form base functionality.
 *
 * Tests validation of form elements against their schema definitions, ensuring
 * proper form generation and schema compliance.
 *
 * @group schema_form
 */
class SchemaConfigFormBaseTest extends SchemaFormTestBase {

  /**
   * A single function to test all the schema forms.
   *
   * We don't need to reinstall the kernel for each test, as the schema
   * definitions are already loaded in the kernel.
   *
   * So, use a single function to speed up the tests.
   */
  public function testConfigForms() {
    // To workaround a core issue
    // https://www.drupal.org/project/drupal/issues/3027240#comment-16109913
    // we need to suppress errors in some function calls.
    // Without this, we get PHP notices when the form is built:
    // @code
    // core/lib/Drupal/Core/Form/FormState.php:1174
    // Undefined array key "#parents"
    // @endcode
    // @todo Remove this when the issue is fixed.
    $originalErrorReporting = error_reporting();

    // Test the Drupal Core's system.performance generated form.
    $schemaData = $this->getSchemaYamlData('system.performance', DRUPAL_ROOT . '/core/modules/system/config/schema/system.schema.yml');
    $form = $this->formBuilder->getForm(CoreSystemPerformanceForm::class);
    $this->checkSchemaItemsInForm($schemaData, $form);

    // Test the MyApiConfigForm.
    $formSchemaName = 'schema_form_test.my_api_config';
    $schemaData = $this->getSchemaYamlData($formSchemaName);
    $schemaData['mapping']['username']['third_party_settings']['schema_form_test'][self::SETTINGS_KEY_OVERRIDDEN_VALUE] = 'john';
    $schemaData['mapping']['password']['third_party_settings']['schema_form_test'][self::SETTINGS_KEY_OVERRIDDEN_VALUE] = self::SETTINGS_KEY_OVERRIDDEN_VALUE_HIDDEN;
    $form = $this->formBuilder->getForm(MyApiConfigForm::class);
    $this->checkSchemaItemsInForm($schemaData, $form);

    // Test the form submission with empty values.
    error_reporting(E_ERROR);
    $formState = $this->createFormState(MyApiConfigForm::class);
    error_reporting($originalErrorReporting);
    $errors = $formState->getErrors();
    $this->assertCount(1, $errors);
    $this->assertEquals($errors['endpoint']->__toString(), 'Endpoint field is required.');

    // Test the form submission with non-valid values.
    error_reporting(E_ERROR);
    $formState = $this->createFormState(MyApiConfigForm::class, $input = [
      'endpoint' => 'example.com',
      'username' => 'john doe',
    ]);
    error_reporting($originalErrorReporting);

    $errors = $formState->getErrors();
    $this->assertCount(2, $errors);
    $this->assertEquals($errors['endpoint']->__toString(), 'The URL <em class="placeholder">example.com</em> is not valid.');
    $this->assertEquals($errors['username']->__toString(), 'No spaces allowed.');

    // Test the form submission and config update.
    $formState = $this->createFormState(MyApiConfigForm::class, $input = [
      'endpoint' => 'http://api.example.com/v1',
      'username' => 'john_doe',
      'password' => 1234,
    ]);
    // Test serialization of the form state.
    try {
      serialize($formState);
    }
    catch (\Exception $e) {
      $this->fail('The form state should be serializable. ' . $e->getMessage());
    }
    $errors = $formState->getErrors();
    $this->assertCount(0, $errors);

    $configBefore = $this->config('schema_form_test.my_api_config');
    $this->assertEquals([], $configBefore->getRawData());
    $this->formBuilder->submitForm(MyApiConfigForm::class, $formState);
    $configAfter = $this->config('schema_form_test.my_api_config');
    $this->assertEquals($input['endpoint'], $configAfter->get('endpoint'));
    $this->assertEquals($input['username'], $configAfter->get('username'));
    $this->assertEquals($input['password'], $configAfter->get('password'));
  }

}
