<?php

declare(strict_types=1);

namespace Drupal\Tests\schema_form\Kernel;

use Drupal\schema_form_test\Form\MyFeedbackForm;
use Drupal\schema_form_test\Form\MyForm;

/**
 * Tests the schema form base functionality.
 *
 * Tests validation of form elements against their schema definitions, ensuring
 * proper form generation and schema compliance.
 *
 * @group schema_form
 */
class SchemaFormBaseTest extends SchemaFormTestBase {

  /**
   * Tests the MyForm class form generation against its schema.
   */
  public function testMyForm() {
    // Test the MyForm generated form.
    $schemaData = $this->getSchemaYamlData('schema_form_test.my_form');
    $form = $this->formBuilder->getForm(MyForm::class);
    $this->checkSchemaItemsInForm($schemaData, $form);

    // Test the demo feedback form.
    $schemaData = $this->getSchemaYamlData('schema_form_test.my_feedback_form');
    $form = $this->formBuilder->getForm(MyFeedbackForm::class);
    $this->checkSchemaItemsInForm($schemaData, $form);

    // Test the form submission with empty values.
    $formState = $this->createFormState(MyFeedbackForm::class);
    $errors = $formState->getErrors();
    $this->assertCount(2, $errors);
    $this->assertEquals($errors['email']->__toString(), "Your email field is required.");
    $this->assertEquals($errors['message']->__toString(), "Message text field is required.");

    // Test the form submission with non-valid values.
    $formState = $this->createFormState(MyFeedbackForm::class, [
      'first_name' => 'John',
      'last_name' => 'Doe',
      'email' => 'foo',
      'message' => 'Great job!',
    ]);
    $errors = $formState->getErrors();
    $this->assertCount(2, $errors);
    $this->assertEquals($errors['email']->__toString(), 'The email address <em class="placeholder">foo</em> is not valid. Use the format user@example.com.');
    $this->assertEquals($errors['message']->__toString(), 'This value is too short. It should have <em class="placeholder">50</em> characters or more.');

    // Test the form submission with valid values.
    $formState = $this->createFormState(MyFeedbackForm::class, [
      'first_name' => 'John',
      'last_name' => 'Doe',
      'email' => 'foo@examile.com',
      'message' => 'Great job! Great job! Great job! Great job! Great job! Great job! Great job! Great job! Great job!',
    ]);
    $errors = $formState->getErrors();
    $this->assertCount(0, $errors);

    // Test serialization of the form state.
    try {
      serialize($formState);
    }
    catch (\Exception $e) {
      $this->fail('The form state should be serializable. ' . $e->getMessage());
    }
  }

}
