<?php

declare(strict_types=1);

namespace Drupal\Tests\schema_form\Kernel;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\KernelTests\KernelTestBase;

/**
 * Base class for schema_form tests.
 */
abstract class SchemaFormTestBase extends KernelTestBase {

  /**
   * The modules to enable for this test.
   *
   * @var string[]
   */
  protected static $modules = [
    'schema_form',
    'schema_form_test',
    'system',
  ];

  const SETTINGS_KEY_OVERRIDDEN_VALUE = 'is_overridden';
  const SETTINGS_KEY_OVERRIDDEN_VALUE_HIDDEN = '__hidden';
  const TRANSLATABLE_PROPERTIES = [
    '#title',
    '#description',
  ];

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $formBuilder;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->formBuilder = $this->container->get('form_builder');
  }

  /**
   * Reads the schema YAML data for a given schema name.
   *
   * @param string $schemaName
   *   The name of the schema to read.
   * @param mixed $filePath
   *   The path to the schema YAML file. If NULL, the default path is used.
   *
   * @return array
   *   The schema data as an associative array.
   */
  protected function getSchemaYamlData(string $schemaName, ?string $filePath = NULL): array {
    static $schemasLoaded = NULL;
    $filePath ??= __DIR__ . '/../../modules/schema_form_test/config/schema/schema_form_test.schema.yml';
    $schemasLoaded[$filePath] ??= Yaml::decode(file_get_contents($filePath));
    return $schemasLoaded[$filePath][$schemaName] ?? NULL;
  }

  /**
   * Recursively checks if form items match their schema definitions.
   *
   * @param array $schemaItems
   *   The schema items to check against the form structure.
   * @param array $form
   *   The form array to validate.
   */
  protected function checkSchemaItemsInForm($schemaItems, $form) {
    foreach ($schemaItems['mapping'] ?? [] as $key => $data) {
      $formItem = $form[$key] ?? NULL;
      $this->assertIsArray($formItem, "Form item $key does not exist.");
      $this->checkSchemaItemInForm($data, $formItem);
    }
  }

  /**
   * Validates a single form item against its schema definition.
   *
   * @param array $schemaItem
   *   The schema item containing the expected values.
   * @param array $form
   *   The form item to validate.
   */
  protected function checkSchemaItemInForm($schemaItem, $form) {
    $this->checkElementValueMatchSchema('#title', $form, $schemaItem);
    $this->checkElementValueMatchSchema('#description', $form, $schemaItem);

    if (isset($schemaItem['mapping'])) {
      $this->checkSchemaItemsInForm($schemaItem, $form);
    }
  }

  /**
   * Checks if the form item value matches the schema.
   *
   * @param string $name
   *   The name of the form item.
   * @param array $element
   *   The form element to check.
   * @param array $schema
   *   The schema definition for the form item.
   *
   * @return mixed
   *   The value from the schema item, or NULL if not found.
   */
  protected function checkElementValueMatchSchema(string $name, array $element, array $schema): mixed {
    // Check if the element value matches the schema definition.
    $value = $this->getFormValueFromSchemaItem($name, $schema);
    if (
      $name === '#description'
      && $valueOverridden = $schema['third_party_settings']['schema_form_test'][self::SETTINGS_KEY_OVERRIDDEN_VALUE] ?? NULL
    ) {
      $valueSuffix = \Drupal::service('schema_form')->getOverriddenMessage($valueOverridden == self::SETTINGS_KEY_OVERRIDDEN_VALUE_HIDDEN ? NULL : $valueOverridden);
      $valueWithSuffix = $value ? $value . ' ' . $valueSuffix : $valueSuffix;
      $this->assertEquals(
        $valueWithSuffix,
        $element[$name],
        "Form item value does not match schema item value."
      );
    }
    elseif ($value !== NULL) {
      $this->assertEquals(
        $value,
        $element[$name],
        "Form item value does not match schema item value."
      );
      if (in_array($name, self::TRANSLATABLE_PROPERTIES)) {
        $this->assertTrue($element[$name] instanceof TranslatableMarkup);
      }
    }
    return $value;
  }

  /**
   * Retrieves the form value from the schema item following the priorities.
   *
   * @param string $name
   *   The name of the form item.
   * @param array $schema
   *   The schema definition for the form item.
   *
   * @return mixed
   *   The value from the schema item, or NULL if not found.
   */
  protected function getFormValueFromSchemaItem(string $name, array $schema): mixed {
    // Priority 1 - look into the third_party_settings for the value.
    if ($value = $schema['third_party_settings']['schema_form'][$name] ?? NULL) {
      return $value;
    }
    if ($value = $schema[$name] ?? NULL) {
      return $value;
    }
    switch ($name) {
      case '#title':
        return $schema['title'] ?? $schema['label'] ?? NULL;

      case '#description':
        return $schema['description'] ?? NULL;
    }
    return NULL;
  }

  /**
   * Creates a form state for the given form class and input values.
   *
   * @param string $formClass
   *   The form class to create the form state for.
   * @param array $input
   *   The input values to set in the form state.
   *
   * @return \Drupal\Core\Form\FormStateInterface
   *   The created form state.
   */
  protected function createFormState(string $formClass, array $input = []): FormStateInterface {
    $formState = new FormState();
    $formId = $this->formBuilder->getFormId($formClass, $formState);
    $formValues = [
      // The submission should have the form_id to handle buttons right.
      'form_id' => $formId,
      ...$input,
    ];
    $formState->setUserInput($formValues);
    $this->formBuilder->buildForm($formClass, $formState);
    return $formState;
  }

}
