<?php

namespace Drupal\schema_metatag_ai\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\schema_metatag_ai\SchemaTypeDiscoveryService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Schema Metatag AI settings.
 */
class SchemaMetatagAiSettingsForm extends ConfigFormBase {

  /**
   * The schema type discovery service.
   *
   * @var \Drupal\schema_metatag_ai\SchemaTypeDiscoveryService
   */
  protected $schemaDiscovery;

  /**
   * Constructs a SchemaMetatagAiSettingsForm object.
   *
   * @param \Drupal\schema_metatag_ai\SchemaTypeDiscoveryService $schema_discovery
   *   The schema type discovery service.
   */
  public function __construct(SchemaTypeDiscoveryService $schema_discovery) {
    $this->schemaDiscovery = $schema_discovery;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('schema_metatag_ai.schema_discovery')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['schema_metatag_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'schema_metatag_ai_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('schema_metatag_ai.settings');

    $form['content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content Types'),
      '#description' => $this->t('Select the content types for which Schema Metatag AI should be enabled.'),
      '#options' => $this->getContentTypeOptions(),
      '#default_value' => $config->get('schema_metatag_ai.content_types') ?: [],
    ];

    $form['ai_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Provider'),
      '#description' => $this->t('Select the AI provider to use for generating schema data.'),
      '#options' => [
        'openai' => $this->t('OpenAI'),
      ],
      '#default_value' => $config->get('schema_metatag_ai.ai_provider') ?: 'openai',
    ];

    $form['ai_model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AI Model'),
      '#description' => $this->t('Specify the AI model to use (e.g., gpt-3.5-turbo, gpt-4).'),
      '#default_value' => $config->get('schema_metatag_ai.ai_model') ?: 'gpt-3.5-turbo',
      '#required' => TRUE,
    ];

    $form['schema_types'] = [
      '#type' => 'details',
      '#title' => $this->t('Schema Type Configuration'),
      '#description' => $this->t('Configure individual schema types, field mappings, and custom prompts. Only enabled schema types from the schema_metatag module are shown.'),
      '#open' => TRUE,
    ];

    // Get available schema types dynamically
    $schema_types = $this->schemaDiscovery->getAvailableSchemaTypes();

    if (empty($schema_types)) {
      $form['schema_types']['no_types'] = [
        '#markup' => '<p>' . $this->t('No schema types available. Please enable schema_metatag submodules (e.g., schema_article, schema_course, etc.).') . '</p>',
      ];
    }
    else {
      foreach ($schema_types as $schema_type_id => $schema_type_info) {
        $schema_config = $config->get("schema_metatag_ai.schema_types.{$schema_type_id}");
        $enabled = $schema_config['enabled'] ?? FALSE;

        $form['schema_types'][$schema_type_id] = [
          '#type' => 'details',
          '#title' => $this->t('@schema_type Schema', ['@schema_type' => $schema_type_info['label']]),
          '#open' => FALSE,
        ];

        $form['schema_types'][$schema_type_id]['info'] = [
          '#markup' => '<p><strong>' . $this->t('Schema.org Type:') . '</strong> ' . $schema_type_info['schema_type'] . '</p>',
        ];

        if (!empty($schema_type_info['url'])) {
          $form['schema_types'][$schema_type_id]['url'] = [
            '#markup' => '<p><strong>' . $this->t('Reference:') . '</strong> <a href="' . $schema_type_info['url'] . '" target="_blank">' . $schema_type_info['url'] . '</a></p>',
          ];
        }

        $form['schema_types'][$schema_type_id]['status'] = [
          '#markup' => '<p><strong>' . $this->t('AI Generation Status:') . '</strong> ' .
                      ($enabled ? $this->t('Enabled') : $this->t('Disabled')) . '</p>',
        ];

        $form['schema_types'][$schema_type_id]['configure'] = [
          '#type' => 'link',
          '#title' => $this->t('Configure @schema_type Schema', ['@schema_type' => $schema_type_info['label']]),
          '#url' => \Drupal\Core\Url::fromRoute('schema_metatag_ai.schema_type_config', ['schema_type' => $schema_type_id]),
          '#attributes' => [
            'class' => ['button'],
          ],
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('schema_metatag_ai.settings');
    
    $content_types = array_filter($form_state->getValue('content_types'));
    
    $config->set('schema_metatag_ai.content_types', array_values($content_types))
           ->set('schema_metatag_ai.ai_provider', $form_state->getValue('ai_provider'))
           ->set('schema_metatag_ai.ai_model', $form_state->getValue('ai_model'))
           ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available content type options.
   *
   * @return array
   *   Array of content type options.
   */
  private function getContentTypeOptions() {
    $content_types = \Drupal::entityTypeManager()->getStorage('node_type')->loadMultiple();
    $options = [];
    
    foreach ($content_types as $content_type) {
      $options[$content_type->id()] = $content_type->label();
    }
    
    return $options;
  }

}