<?php

namespace Drupal\schema_metatag_ai\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for individual Schema type settings.
 */
class SchemaTypeConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['schema_metatag_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'schema_type_config_form';
  }

  /**
   * Get the page title for schema type configuration.
   *
   * @param string $schema_type
   *   The schema type.
   *
   * @return string
   *   The page title.
   */
  public static function getTitle($schema_type) {
    return t('Configure @schema_type Schema', ['@schema_type' => ucfirst($schema_type)]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $schema_type = NULL) {
    $config = $this->config('schema_metatag_ai.settings');
    $schema_config = $config->get("schema_metatag_ai.schema_types.{$schema_type}") ?: [];

    if (!$schema_type) {
      $this->messenger()->addError($this->t('Schema type not specified.'));
      return $form;
    }

    $form['schema_type'] = [
      '#type' => 'value',
      '#value' => $schema_type,
    ];

    $form['info'] = [
      '#markup' => '<h2>' . $this->t('Configure @schema_type Schema', ['@schema_type' => ucfirst($schema_type)]) . '</h2>',
    ];

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable @schema_type Schema Generation', ['@schema_type' => ucfirst($schema_type)]),
      '#default_value' => $schema_config['enabled'] ?? TRUE,
    ];

    $form['prompt_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt Template'),
      '#description' => $this->t('Use [FIELD:field_name] placeholders to reference Drupal fields. Leave empty to use default template.'),
      '#default_value' => $schema_config['prompt_template'] ?? '',
      '#rows' => 15,
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['field_mappings'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Mappings'),
      '#description' => $this->t('Map Drupal fields to schema properties for data extraction.'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $field_mappings = $schema_config['field_mappings'] ?? [];
    $schema_field_mappings = $schema_config['schema_field_mappings'] ?? [];

    // Common schema fields
    $common_fields = ['name', 'description', 'keywords'];
    
    // Schema-specific fields
    $schema_specific_fields = $this->getSchemaSpecificFields($schema_type);
    
    $all_fields = array_merge($common_fields, $schema_specific_fields);

    foreach ($all_fields as $field) {
      // Auto-generate the schema form field ID
      $auto_field_id = $this->generateSchemaFieldId($field, $schema_type);

      $form['field_mappings'][$field] = [
        '#type' => 'details',
        '#title' => $this->t('@field Field', ['@field' => ucfirst($field)]),
        '#open' => FALSE,
      ];

      $form['field_mappings'][$field]["drupal_field_{$field}"] = [
        '#type' => 'textfield',
        '#title' => $this->t('Drupal Field Name'),
        '#description' => $this->t('The Drupal field to extract data from (e.g., title, body, field_ucas_code).'),
        '#default_value' => $field_mappings[$field] ?? '',
      ];

      $form['field_mappings'][$field]["schema_field_{$field}"] = [
        '#type' => 'textfield',
        '#title' => $this->t('Schema Form Field ID'),
        '#default_value' => $schema_field_mappings[$field] ?? $auto_field_id,
        '#attributes' => [
          'placeholder' => $auto_field_id,
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Generate schema form field ID from field name and schema type.
   *
   * @param string $field_name
   *   The field name (e.g., 'name', 'courseCode').
   * @param string $schema_type
   *   The schema type (e.g., 'course', 'article').
   *
   * @return string
   *   The generated schema form field ID (e.g., 'schema-course-name').
   */
  private function generateSchemaFieldId($field_name, $schema_type) {
    // Convert camelCase to kebab-case
    $kebab_field = preg_replace('/([a-z])([A-Z])/', '$1-$2', $field_name);
    $kebab_field = strtolower($kebab_field);

    // Format: schema-{type}-{field}
    return "schema-{$schema_type}-{$kebab_field}";
  }

  /**
   * Get schema-specific fields based on schema type.
   *
   * This method provides common schema fields that can be configured.
   * Users can add additional fields as needed in the configuration.
   *
   * @param string $schema_type
   *   The schema type.
   *
   * @return array
   *   Array of common schema-specific field names.
   */
  private function getSchemaSpecificFields($schema_type) {
    // Return a generic set of common fields that apply to most schema types
    // The generic field mapper will handle any additional fields automatically
    $common_schema_fields = [
      'headline',
      'author',
      'publisher',
      'datePublished',
      'dateModified',
      'image',
      'url',
    ];

    // Schema-specific fields (extensible - these are just common examples)
    $specific_fields_map = [
      'course' => ['courseCode', 'educationalLevel', 'teaches', 'timeRequired', 'provider'],
      'product' => ['sku', 'brand', 'price', 'availability', 'offers'],
      'event' => ['startDate', 'endDate', 'location', 'organizer'],
      'person' => ['jobTitle', 'worksFor', 'birthDate', 'email', 'telephone'],
      'organization' => ['address', 'telephone', 'email', 'logo'],
      'place' => ['address', 'telephone', 'openingHours', 'geo'],
      'recipe' => ['recipeIngredient', 'recipeInstructions', 'cookTime', 'prepTime'],
      'book' => ['isbn', 'numberOfPages', 'bookFormat'],
      'movie' => ['director', 'actor', 'genre', 'duration'],
      'review' => ['reviewRating', 'itemReviewed', 'reviewBody'],
      'article' => ['articleBody', 'wordCount'],
      'web_page' => ['mainEntity', 'breadcrumb'],
      'how_to' => ['step', 'totalTime', 'tool', 'supply'],
      'qa_page' => ['mainEntity'],
      'job_posting' => ['baseSalary', 'employmentType', 'hiringOrganization', 'jobLocation'],
      'video_object' => ['contentUrl', 'thumbnailUrl', 'uploadDate', 'duration'],
    ];

    return $specific_fields_map[$schema_type] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('schema_metatag_ai.settings');
    $schema_type = $form_state->getValue('schema_type');

    $field_mappings = [];
    $schema_field_mappings = [];

    // Extract field mappings from form values
    foreach ($form_state->getValues() as $key => $value) {
      if (strpos($key, 'drupal_field_') === 0) {
        $field_name = str_replace('drupal_field_', '', $key);
        if (!empty($value)) {
          $field_mappings[$field_name] = $value;
        }
      }
      if (strpos($key, 'schema_field_') === 0) {
        $field_name = str_replace('schema_field_', '', $key);
        if (!empty($value)) {
          $schema_field_mappings[$field_name] = $value;
        }
      }
    }

    $schema_config = [
      'enabled' => $form_state->getValue('enabled'),
      'prompt_template' => $form_state->getValue('prompt_template'),
      'field_mappings' => $field_mappings,
      'schema_field_mappings' => $schema_field_mappings,
    ];

    $config->set("schema_metatag_ai.schema_types.{$schema_type}", $schema_config)->save();

    parent::submitForm($form, $form_state);
  }

}