<?php

namespace Drupal\schema_metatag_ai\Plugin;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;

/**
 * Main class that generates schema metatag using AI.
 */
class GenerateSchemaMetatag {

  /**
   * Stores error messages.
   *
   * @var array
   */
  private $errors = [];

  /**
   * Stores Drupal logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private $logger;

  /**
   * Stores Drupal configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private $config;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * AI Provider Manager service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * Alias manager service.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Schema type discovery service.
   *
   * @var \Drupal\schema_metatag_ai\SchemaTypeDiscoveryService
   */
  protected $schemaDiscovery;

  /**
   * GenerateSchemaMetatag constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider manager service.
   * @param \Drupal\path_alias\AliasManagerInterface $alias_manager
   *   The alias manager service.
   * @param \Drupal\schema_metatag_ai\SchemaTypeDiscoveryService $schema_discovery
   *   The schema type discovery service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, AiProviderPluginManager $ai_provider_manager, AliasManagerInterface $alias_manager = NULL, $schema_discovery = NULL) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->aiProviderManager = $ai_provider_manager;
    $this->aliasManager = $alias_manager ?: \Drupal::service('path_alias.manager');
    $this->schemaDiscovery = $schema_discovery ?: \Drupal::service('schema_metatag_ai.schema_discovery');
    $this->config = $config_factory->get('schema_metatag_ai.settings');
    $this->logger = $logger_factory->get('schema_metatag_ai');
  }

  /**
   * Generate schema metatag data using AI.
   *
   * @param array $field_data
   *   Array of field data with field names as keys.
   * @param string $content_type
   *   The content type (e.g., 'article', 'product', 'event').
   * @param object $entity
   *   The entity being processed.
   *
   * @return array|null
   *   Generated schema data or null on failure.
   */
  public function generate($field_data, $content_type = 'article', $entity = NULL) {
    $this->errors = [];

    if (empty($field_data)) {
      $this->errors[] = 'Field data is empty.';
      return NULL;
    }

    try {
      $prompt = $this->buildPrompt($field_data, $content_type, $entity);
      
      // Create AI chat input
      $messages = new ChatInput([
        new ChatMessage('user', $prompt),
      ]);
      
      // Get AI provider settings
      $ai_provider = $this->config->get('schema_metatag_ai.ai_provider') ?: 'openai';
      $ai_model = $this->config->get('schema_metatag_ai.ai_model') ?: 'gpt-3.5-turbo';
      
      // Use the AI service to generate schema data
      $provider = $this->aiProviderManager->createInstance($ai_provider);
      $response = $provider->chat($messages, $ai_model);

      if (!$response) {
        $this->errors[] = 'Failed to get response from AI service.';
        return NULL;
      }

      $generated_data = $this->parseAiResponse($response, $entity);
      
      if (!$generated_data) {
        $this->errors[] = 'Failed to parse AI response.';
        return NULL;
      }

      return $generated_data;

    } catch (\Exception $e) {
      $this->errors[] = 'Error generating schema data: ' . $e->getMessage();
      $this->logger->error('Schema metatag AI generation failed: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Build the AI prompt for schema generation.
   *
   * @param array $field_data
   *   Array of field data with field names as keys.
   * @param string $content_type
   *   The content type.
   * @param object $entity
   *   The entity being processed.
   *
   * @return string
   *   The prompt for AI generation.
   */
  private function buildPrompt($field_data, $content_type, $entity = NULL) {
    // Get schema type info dynamically
    $schema_type_info = $this->schemaDiscovery->getSchemaType($content_type);

    if (!$schema_type_info) {
      // Fallback to Article if schema type not found
      $schema_type_info = $this->schemaDiscovery->getSchemaType('article');
      if (!$schema_type_info) {
        // Ultimate fallback
        return $this->buildDefaultPrompt($field_data, 'Article');
      }
    }

    $schema_type = $schema_type_info['schema_type'];
    $schema_url = $schema_type_info['url'];

    // Get schema type configuration
    $schema_config = $this->config->get("schema_metatag_ai.schema_types.{$content_type}");

    // Check if custom prompt template exists and is enabled
    if ($schema_config && isset($schema_config['enabled']) && $schema_config['enabled'] && !empty($schema_config['prompt_template'])) {
      // Use custom prompt template
      $prompt_template = $schema_config['prompt_template'];
      $field_mappings = $schema_config['field_mappings'] ?? [];

      // Replace field placeholders in the prompt template
      $prompt = $this->replacePlaceholders($prompt_template, $field_data, $field_mappings, $entity);

      // Add the field data as content for analysis
      $content_for_analysis = $this->buildContentForAnalysis($field_data, $field_mappings);
      $prompt .= "\n\nContent to analyze:\n{$content_for_analysis}";

      return $prompt;
    }

    // Use generic prompt that references Schema.org
    return $this->buildGenericPrompt($field_data, $schema_type, $schema_url);
  }

  /**
   * Replace placeholders in prompt template with actual field values.
   *
   * @param string $template
   *   The prompt template.
   * @param array $field_data
   *   Array of field data.
   * @param array $field_mappings
   *   Field mapping configuration.
   * @param object $entity
   *   The entity being processed.
   *
   * @return string
   *   The processed prompt.
   */
  private function replacePlaceholders($template, $field_data, $field_mappings, $entity = NULL) {
    $processed_template = $template;

    // Replace [FIELD:field_name] placeholders
    foreach ($field_mappings as $schema_field => $drupal_field) {
      $placeholder = "[FIELD:{$drupal_field}]";
      $field_value = $field_data[$drupal_field] ?? '';
      
      // If we have an entity, try to get the field value from it
      if ($entity && $entity->hasField($drupal_field)) {
        $field_value = $this->getFieldValue($entity, $drupal_field);
      }
      
      $processed_template = str_replace($placeholder, $field_value, $processed_template);
    }

    return $processed_template;
  }

  /**
   * Get field value from entity.
   *
   * @param object $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   The field value.
   */
  private function getFieldValue($entity, $field_name) {
    if (!$entity->hasField($field_name)) {
      return '';
    }

    $field = $entity->get($field_name);
    if ($field->isEmpty()) {
      return '';
    }

    $field_type = $field->getFieldDefinition()->getType();
    
    switch ($field_type) {
      case 'string':
      case 'string_long':
      case 'text':
      case 'text_long':
      case 'text_with_summary':
        return $field->value ?? '';
        
      case 'entity_reference':
        $referenced_entity = $field->entity;
        if ($referenced_entity) {
          return $referenced_entity->label();
        }
        return '';
        
      case 'list_string':
        return $field->value ?? '';
        
      default:
        return (string) $field->value ?? '';
    }
  }

  /**
   * Build content for analysis from field data.
   *
   * @param array $field_data
   *   Array of field data.
   * @param array $field_mappings
   *   Field mapping configuration.
   *
   * @return string
   *   Content for AI analysis.
   */
  private function buildContentForAnalysis($field_data, $field_mappings) {
    $content_parts = [];

    // If field mappings are configured, use them
    if (!empty($field_mappings)) {
      foreach ($field_mappings as $schema_field => $drupal_field) {
        if (isset($field_data[$drupal_field]) && !empty($field_data[$drupal_field])) {
          $content_parts[] = "{$schema_field}: {$field_data[$drupal_field]}";
        }
      }
    }
    else {
      // No mappings configured - include ALL field data
      foreach ($field_data as $field_name => $field_value) {
        if (!empty($field_value)) {
          // Clean up field name for display
          $clean_name = ucfirst(str_replace(['field_', '_'], ['', ' '], $field_name));
          $content_parts[] = "{$clean_name}: {$field_value}";
        }
      }
    }

    return implode("\n", $content_parts);
  }

  /**
   * Build generic prompt that references Schema.org definitions.
   *
   * @param array $field_data
   *   Array of field data.
   * @param string $schema_type
   *   The Schema.org type (e.g., 'Article', 'Course').
   * @param string $schema_url
   *   The Schema.org URL for this type.
   *
   * @return string
   *   Generic prompt.
   */
  private function buildGenericPrompt($field_data, $schema_type, $schema_url = '') {
    $prompt = "You are a Schema.org structured data expert. Generate valid Schema.org JSON-LD markup for a {$schema_type} type.\n\n";

    if (!empty($schema_url)) {
      $prompt .= "Reference the official Schema.org definition at: {$schema_url}\n\n";
    }

    $prompt .= "Requirements:\n";
    $prompt .= "1. Return ONLY a valid JSON object (not JSON-LD with @context)\n";
    $prompt .= "2. Include the @type field set to '{$schema_type}'\n";
    $prompt .= "3. Populate all relevant and required properties for {$schema_type} based on the content provided\n";
    $prompt .= "4. Use proper Schema.org property names and data types\n";
    $prompt .= "5. For nested objects (like author, publisher, provider), use the format: {\"@type\": \"ObjectType\", \"name\": \"value\"}\n";
    $prompt .= "6. Do not include @context or @id fields - these will be added automatically\n";
    $prompt .= "7. Only return the JSON object, no additional text or explanation\n\n";

    // Build content from field data
    $content_parts = [];
    foreach ($field_data as $field_name => $field_value) {
      if (!empty($field_value) && is_string($field_value)) {
        $clean_field_name = ucfirst(str_replace(['field_', '_'], ['', ' '], $field_name));
        $content_parts[] = "{$clean_field_name}: {$field_value}";
      }
    }

    $content = implode("\n", $content_parts);
    $prompt .= "Content to analyze:\n{$content}";

    return $prompt;
  }

  /**
   * Build default prompt when no custom configuration exists.
   *
   * @param array $field_data
   *   Array of field data.
   * @param string $schema_type
   *   The schema type.
   *
   * @return string
   *   Default prompt.
   */
  private function buildDefaultPrompt($field_data, $schema_type) {
    return $this->buildGenericPrompt($field_data, $schema_type);
  }

  /**
   * Parse the AI response and extract schema data.
   *
   * @param mixed $response
   *   The AI response.
   * @param object $entity
   *   The entity being processed.
   *
   * @return array|null
   *   Parsed schema data or null on failure.
   */
  private function parseAiResponse($response, $entity = NULL) {
    try {
      // Extract the text content from the AI response
      $response_text = '';
      if (is_object($response) && method_exists($response, 'getNormalized')) {
        $normalized = $response->getNormalized();
        if (method_exists($normalized, 'getText')) {
          $response_text = $normalized->getText();
        }
      } elseif (is_object($response) && method_exists($response, 'getText')) {
        $response_text = $response->getText();
      } elseif (is_array($response) && isset($response['text'])) {
        $response_text = $response['text'];
      } elseif (is_string($response)) {
        $response_text = $response;
      }

      if (empty($response_text)) {
        return NULL;
      }

      // Try to extract JSON from the response
      $response_text = trim($response_text);
      
      // Remove any surrounding code blocks or markdown
      $response_text = preg_replace('/^```json\s*/', '', $response_text);
      $response_text = preg_replace('/\s*```$/', '', $response_text);
      
      // Decode JSON
      $schema_data = json_decode($response_text, TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->errors[] = 'Invalid JSON in AI response: ' . json_last_error_msg();
        return NULL;
      }

      // Enhance schema data with @id and @type
      $schema_data = $this->enhanceSchemaData($schema_data, $entity);

      return $schema_data;

    } catch (\Exception $e) {
      $this->errors[] = 'Error parsing AI response: ' . $e->getMessage();
      return NULL;
    }
  }

  /**
   * Enhance schema data with @id and @type fields.
   *
   * @param array $schema_data
   *   The schema data array.
   * @param object $entity
   *   The entity being processed.
   *
   * @return array
   *   Enhanced schema data.
   */
  private function enhanceSchemaData(array $schema_data, $entity = NULL) {
    if (!$entity) {
      return $schema_data;
    }

    // Ensure @context is set
    if (!isset($schema_data['@context'])) {
      $schema_data['@context'] = 'https://schema.org';
    }

    // Ensure @type is properly set if not already present
    if (!isset($schema_data['@type']) && isset($schema_data['type'])) {
      $schema_data['@type'] = $schema_data['type'];
      unset($schema_data['type']);
    }

    // Generate @id with full URL using alias if available
    if (!isset($schema_data['@id'])) {
      $schema_data['@id'] = $this->generateEntityUrl($entity);
    }

    return $schema_data;
  }

  /**
   * Generate full URL for entity with alias if available.
   *
   * @param object $entity
   *   The entity.
   *
   * @return string
   *   The full URL for the entity.
   */
  private function generateEntityUrl($entity) {
    if (!$entity || !method_exists($entity, 'toUrl')) {
      return '';
    }

    try {
      // Get the canonical URL for the entity
      $url = $entity->toUrl('canonical', ['absolute' => TRUE]);
      $full_url = $url->toString();
      
      // Check if there's an alias available
      if ($this->aliasManager && method_exists($entity, 'id') && method_exists($entity, 'getEntityTypeId')) {
        $system_path = '/' . $entity->getEntityTypeId() . '/' . $entity->id();
        $alias = $this->aliasManager->getAliasByPath($system_path);
        
        // If an alias exists and is different from the system path
        if ($alias && $alias !== $system_path) {
          // Replace the system path with the alias in the full URL
          $base_url = \Drupal::request()->getSchemeAndHttpHost();
          $full_url = $base_url . $alias;
        }
      }
      
      return $full_url;
    } catch (\Exception $e) {
      $this->logger->warning('Failed to generate entity URL: @error', ['@error' => $e->getMessage()]);
      return '';
    }
  }

  /**
   * Check if there are any errors.
   *
   * @return bool
   *   TRUE if there are errors, FALSE otherwise.
   */
  public function hasErrors() {
    return !empty($this->errors);
  }

  /**
   * Get error messages.
   *
   * @return array
   *   Array of error messages.
   */
  public function getErrors() {
    return $this->errors;
  }

}