<?php

namespace Drupal\schema_metatag_ai;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\MessageCommand;

/**
 * Service for mapping AI-generated schema data to Drupal form fields.
 */
class SchemaFieldMapperService {

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a SchemaFieldMapperService object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Map AI response data to schema form fields.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   * @param array $result
   *   The AI generation result.
   * @param string $schema_type
   *   The schema type (e.g., 'article', 'course').
   * @param string $elem_id
   *   The element ID.
   * @param array $custom_mappings
   *   Optional custom field mappings from configuration.
   */
  public function mapFieldsToForm(AjaxResponse $response, array $result, $schema_type, $elem_id, array $custom_mappings = []) {
    // Always populate @type and @id fields first
    $this->populateCoreFields($response, $result, $schema_type, $elem_id);

    // If custom mappings are provided, use them
    if (!empty($custom_mappings)) {
      $this->applyCustomMappings($response, $result, $schema_type, $elem_id, $custom_mappings);
      return;
    }

    // Otherwise, use generic mapping
    $this->applyGenericMapping($response, $result, $schema_type, $elem_id);
  }

  /**
   * Populate core schema fields (@type and @id).
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   * @param array $result
   *   The AI generation result.
   * @param string $schema_type
   *   The schema type.
   * @param string $elem_id
   *   The element ID.
   */
  protected function populateCoreFields(AjaxResponse $response, array $result, $schema_type, $elem_id) {
    // Map schema types to their Schema.org @type values
    $type_mapping = [
      'article' => 'Article',
      'course' => 'Course',
      'product' => 'Product',
      'event' => 'Event',
      'person' => 'Person',
      'organization' => 'Organization',
      'place' => 'Place',
      'recipe' => 'Recipe',
      'book' => 'Book',
      'movie' => 'Movie',
      'review' => 'Review',
      'web_page' => 'WebPage',
      'web_site' => 'WebSite',
      'how_to' => 'HowTo',
      'qa_page' => 'QAPage',
      'service' => 'Service',
      'job_posting' => 'JobPosting',
      'video_object' => 'VideoObject',
      'image_object' => 'ImageObject',
      'item_list' => 'ItemList',
      'special_announcement' => 'SpecialAnnouncement',
    ];

    // Set @type field
    $type_value = $result['@type'] ?? $type_mapping[$schema_type] ?? ucfirst(str_replace('_', '', $schema_type));
    $type_selector = "#edit-{$elem_id}-0-schema-{$schema_type}-schema-{$schema_type}-type";

    $response->addCommand(new InvokeCommand($type_selector, 'val', [$type_value]));
    $response->addCommand(new InvokeCommand($type_selector, 'trigger', ['change']));

    // Set @id field if present
    if (!empty($result['@id'])) {
      $id_selector = "#edit-{$elem_id}-0-schema-{$schema_type}-schema-{$schema_type}-id";
      $response->addCommand(new InvokeCommand($id_selector, 'val', [$result['@id']]));
    }
  }

  /**
   * Apply custom field mappings from configuration.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   * @param array $result
   *   The AI generation result.
   * @param string $schema_type
   *   The schema type.
   * @param string $elem_id
   *   The element ID.
   * @param array $custom_mappings
   *   Custom field mappings.
   */
  protected function applyCustomMappings(AjaxResponse $response, array $result, $schema_type, $elem_id, array $custom_mappings) {
    foreach ($custom_mappings as $result_field => $form_field_id) {
      if (!empty($result[$result_field])) {
        $field_selector = "#edit-{$elem_id}-0-schema-{$schema_type}-{$form_field_id}";

        // Handle nested objects (like author, publisher, provider)
        if (is_array($result[$result_field])) {
          if (isset($result[$result_field]['name'])) {
            $response->addCommand(new InvokeCommand($field_selector, 'val', [$result[$result_field]['name']]));
          }
        }
        else {
          $response->addCommand(new InvokeCommand($field_selector, 'val', [$result[$result_field]]));
        }
      }
    }
  }

  /**
   * Apply generic field mapping for all schema properties.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   * @param array $result
   *   The AI generation result.
   * @param string $schema_type
   *   The schema type.
   * @param string $elem_id
   *   The element ID.
   */
  protected function applyGenericMapping(AjaxResponse $response, array $result, $schema_type, $elem_id) {
    // Skip @context, @type, and @id as they're handled separately
    $skip_fields = ['@context', '@type', '@id'];

    $selectors_tried = [];

    foreach ($result as $property => $value) {
      if (in_array($property, $skip_fields) || empty($value)) {
        continue;
      }

      // Convert property name to multiple possible field name formats
      $possible_selectors = $this->generatePossibleSelectors($property, $schema_type, $elem_id);

      $value_to_set = $this->prepareValue($value);
      if ($value_to_set === NULL) {
        continue;
      }

      // Try each possible selector
      foreach ($possible_selectors as $selector) {
        $response->addCommand(new InvokeCommand($selector, 'val', [$value_to_set]));
        $selectors_tried[] = "{$property} => {$selector}";
      }
    }

    // Add debug message with selectors
    if (!empty($selectors_tried)) {
      $debug_msg = 'Selectors used for ' . $schema_type . ': ' . implode(', ', $selectors_tried);
      \Drupal::logger('schema_metatag_ai')->notice($debug_msg);
    }
  }

  /**
   * Generate multiple possible selectors for a property.
   *
   * @param string $property
   *   The Schema.org property name.
   * @param string $schema_type
   *   The schema type.
   * @param string $elem_id
   *   The element ID.
   *
   * @return array
   *   Array of possible selectors to try.
   */
  protected function generatePossibleSelectors($property, $schema_type, $elem_id) {
    $selectors = [];

    // Convert property name to form field format
    $form_field_name = $this->convertPropertyToFieldName($property);

    // Pattern 1: Standard schema metatag pattern
    // e.g., #edit-field-metatags-0-schema-article-schema-article-name
    $selectors[] = "#edit-{$elem_id}-0-schema-{$schema_type}-schema-{$schema_type}-{$form_field_name}";

    // Pattern 2: Simplified pattern (just property name)
    // e.g., #edit-field-metatags-0-schema-article-name
    $selectors[] = "#edit-{$elem_id}-0-schema-{$schema_type}-{$form_field_name}";

    // Pattern 3: With underscores instead of hyphens in property name
    $underscore_name = str_replace('-', '_', $form_field_name);
    $selectors[] = "#edit-{$elem_id}-0-schema-{$schema_type}-schema-{$schema_type}-{$underscore_name}";

    // Pattern 4: Direct property name (for common fields)
    $selectors[] = "#edit-{$elem_id}-0-schema-{$schema_type}-{$property}";

    return $selectors;
  }

  /**
   * Prepare value for setting in form field.
   *
   * @param mixed $value
   *   The value to prepare.
   *
   * @return string|null
   *   The prepared value or NULL if it can't be converted.
   */
  protected function prepareValue($value) {
    // Handle different value types
    if (is_array($value)) {
      // For nested objects, try to extract name property
      if (isset($value['name'])) {
        return (string) $value['name'];
      }
      // For nested objects with @type (like Person, Organization)
      if (isset($value['@type'])) {
        // Try to extract a useful value
        foreach (['name', 'email', 'url', 'telephone'] as $key) {
          if (isset($value[$key])) {
            return (string) $value[$key];
          }
        }
        // If it's a Person or Organization, serialize as JSON
        return json_encode($value);
      }
      // For arrays of values, join them
      if (isset($value[0]) && is_string($value[0])) {
        return implode(', ', $value);
      }
      // Can't convert this array
      return NULL;
    }

    if (is_string($value) || is_numeric($value)) {
      return (string) $value;
    }

    return NULL;
  }

  /**
   * Convert Schema.org property name to form field name.
   *
   * @param string $property
   *   The Schema.org property name (e.g., "courseCode", "datePublished").
   *
   * @return string
   *   The form field name (e.g., "course-code", "date-published").
   */
  protected function convertPropertyToFieldName($property) {
    // Convert camelCase to kebab-case
    $field_name = preg_replace('/([a-z])([A-Z])/', '$1-$2', $property);
    return strtolower($field_name);
  }

}
