<?php

namespace Drupal\schema_metatag_ai;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\metatag\MetatagGroupPluginManager;

/**
 * Service for discovering available schema types from schema_metatag module.
 */
class SchemaTypeDiscoveryService {

  /**
   * The metatag group plugin manager.
   *
   * @var \Drupal\metatag\MetatagGroupPluginManager
   */
  protected $metatagGroupManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a SchemaTypeDiscoveryService object.
   *
   * @param \Drupal\metatag\MetatagGroupPluginManager $metatag_group_manager
   *   The metatag group plugin manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(MetatagGroupPluginManager $metatag_group_manager, ModuleHandlerInterface $module_handler, LoggerChannelFactoryInterface $logger_factory) {
    $this->metatagGroupManager = $metatag_group_manager;
    $this->moduleHandler = $module_handler;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Get all available schema types from enabled schema_metatag modules.
   *
   * @return array
   *   Array of schema types with keys:
   *   - id: The schema type ID (e.g., 'article', 'course')
   *   - label: The human-readable label
   *   - group_id: The metatag group ID (e.g., 'schema_article')
   *   - schema_type: The Schema.org type (e.g., 'Article', 'Course')
   *   - url: The Schema.org URL for this type
   */
  public function getAvailableSchemaTypes() {
    $schema_types = [];

    // Get all metatag group definitions
    $group_definitions = $this->metatagGroupManager->getDefinitions();

    foreach ($group_definitions as $group_id => $definition) {
      // Only include schema.org groups (they start with 'schema_')
      if (strpos($group_id, 'schema_') === 0 && $group_id !== 'schema_metatag') {
        // Extract the schema type from group ID (e.g., 'schema_article' => 'article')
        $schema_type_id = str_replace('schema_', '', $group_id);

        // Get the Schema.org type from the label (e.g., "Schema.org: Article" => "Article")
        $label = (string) $definition['label'];
        $schema_org_type = trim(str_replace('Schema.org:', '', $label));

        // Extract Schema.org URL from description if available
        $schema_url = $this->extractSchemaUrl($definition);

        $schema_types[$schema_type_id] = [
          'id' => $schema_type_id,
          'label' => $schema_org_type,
          'group_id' => $group_id,
          'schema_type' => $schema_org_type,
          'url' => $schema_url,
        ];
      }
    }

    return $schema_types;
  }

  /**
   * Extract Schema.org URL from group definition.
   *
   * @param array $definition
   *   The group definition.
   *
   * @return string
   *   The Schema.org URL or empty string.
   */
  protected function extractSchemaUrl($definition) {
    if (isset($definition['description'])) {
      $description = (string) $definition['description'];

      // Try to extract URL from description arguments
      if (isset($definition['description']->getArguments()[':url'])) {
        return $definition['description']->getArguments()[':url'];
      }

      // Fallback: try to extract from the description text
      if (preg_match('/https:\/\/schema\.org\/(\w+)/', $description, $matches)) {
        return $matches[0];
      }
    }

    return '';
  }

  /**
   * Get schema type information by ID.
   *
   * @param string $schema_type_id
   *   The schema type ID (e.g., 'article', 'course').
   *
   * @return array|null
   *   Schema type information or NULL if not found.
   */
  public function getSchemaType($schema_type_id) {
    $schema_types = $this->getAvailableSchemaTypes();
    return $schema_types[$schema_type_id] ?? NULL;
  }

  /**
   * Get schema type options for form select elements.
   *
   * @return array
   *   Array of options with schema type ID as key and label as value.
   */
  public function getSchemaTypeOptions() {
    $schema_types = $this->getAvailableSchemaTypes();
    $options = [];

    foreach ($schema_types as $type_id => $type_info) {
      $options[$type_id] = $type_info['label'];
    }

    // Sort alphabetically by label
    asort($options);

    return $options;
  }

  /**
   * Check if a schema type is available.
   *
   * @param string $schema_type_id
   *   The schema type ID to check.
   *
   * @return bool
   *   TRUE if the schema type is available, FALSE otherwise.
   */
  public function isSchemaTypeAvailable($schema_type_id) {
    $schema_types = $this->getAvailableSchemaTypes();
    return isset($schema_types[$schema_type_id]);
  }

}
